const mainDoc = document.querySelector("html");
const mainElement = document.querySelector("body.dashboard main");

let notification_button = document.querySelector(".notification-bell");
let notification_closeButton = document.querySelector(
  ".notification .title .close-button"
);

let profile_section = document.querySelector(".profile > p");
let notification_items = document.querySelectorAll(".notification-items li");

let active_tab = "home";
let restaurantID = "543219876";
let deliveryPage = "in-house";

let restaurantData;
let menuData;
let deliveryPartnersData;
let inHouseDeliveryPartnersData;
let restaurantReviews;
const editDishForm = document.getElementById("edit-form");

let all_orders = [];

let order_sample_names = [
  "Salis James",
  "John Wick",
  "James Bond",
  "Steve Jobs",
  "Michael Jordan",
  "Moses Irin",
  "Esan Lola",
  "Sadeeq Yusuf",
  "Mikel Obi",
  "VDM Bola",
  "John Toheeb",
  "Falz Davido",
];
let order_sample_address = [
  "29 Westheimer Rd. Santa Ana, Illinois 85484",
  "972 Westheimer Rd. Santa Ana, Illinois 85484",
  "297 Westheimer Rd. Santa Ana, Illinois 85484",
  "97 Westheimer Rd. Santa Ana, Illinois 85484",
  "272 Westheimer Rd. Santa Ana, Illinois 85484",
  "292 Westheimer Rd. Santa Ana, Illinois 85484",
];

let order_sample_types = ["Delivery", "Takeaway", "Dine in"];

let order_icons = [
  `
   <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path d="M13.25 7.75C13.1737 7.75 13.0981 7.75375 13.0238 7.76L10.9525 2.23688C10.8989 2.09384 10.8029 1.97058 10.6774 1.88357C10.5519 1.79657 10.4027 1.74997 10.25 1.75H8.25C8.05109 1.75 7.86032 1.82902 7.71967 1.96967C7.57902 2.11032 7.5 2.30109 7.5 2.5C7.5 2.69891 7.57902 2.88968 7.71967 3.03033C7.86032 3.17098 8.05109 3.25 8.25 3.25H9.73L11.6194 8.2875C11.201 8.59612 10.8781 9.01633 10.6875 9.5H8.8625L7.4525 5.73688C7.39891 5.59384 7.30293 5.47058 7.17739 5.38357C7.05185 5.29657 6.90274 5.24996 6.75 5.25H1.75C1.55109 5.25 1.36032 5.32902 1.21967 5.46967C1.07902 5.61032 1 5.80109 1 6C1 6.19891 1.07902 6.38968 1.21967 6.53033C1.36032 6.67098 1.55109 6.75 1.75 6.75H2V6.9325C1.39687 7.25182 0.892063 7.72931 0.53972 8.31376C0.187377 8.89821 0.000798578 9.56756 0 10.25C0 10.4489 0.0790176 10.6397 0.21967 10.7803C0.360322 10.921 0.551088 11 0.75 11H1.04625C1.16354 11.6314 1.49812 12.2019 1.99197 12.6124C2.48581 13.023 3.10778 13.2478 3.75 13.2478C4.39222 13.2478 5.01419 13.023 5.50803 12.6124C6.00188 12.2019 6.33646 11.6314 6.45375 11H10.5463C10.6413 11.5138 10.8807 11.9899 11.2365 12.3726C11.5923 12.7553 12.0498 13.0286 12.5553 13.1607C13.0609 13.2928 13.5936 13.2781 14.0911 13.1183C14.5886 12.9586 15.0303 12.6604 15.3645 12.2587C15.6987 11.8571 15.9115 11.3685 15.978 10.8502C16.0446 10.3319 15.9621 9.80543 15.7402 9.33235C15.5183 8.85927 15.1663 8.45918 14.7253 8.17895C14.2843 7.89871 13.7725 7.74992 13.25 7.75ZM3 8.12813C3.14629 8.0764 3.27294 7.98058 3.36249 7.85387C3.45204 7.72716 3.50008 7.57579 3.5 7.42063V6.75H6.23L7.26125 9.5H1.62875C1.74053 9.18354 1.92168 8.89609 2.15895 8.65871C2.39622 8.42133 2.68359 8.24005 3 8.12813ZM3.75 11.75C3.50689 11.7499 3.26909 11.6789 3.06572 11.5456C2.86235 11.4124 2.70224 11.2228 2.605 11H4.895C4.79776 11.2228 4.63765 11.4124 4.43428 11.5456C4.23091 11.6789 3.99311 11.7499 3.75 11.75ZM13.25 11.75C13.0028 11.75 12.7611 11.6767 12.5555 11.5393C12.35 11.402 12.1898 11.2068 12.0952 10.9784C12.0005 10.7499 11.9758 10.4986 12.024 10.2561C12.0722 10.0137 12.1913 9.79093 12.3661 9.61612C12.5409 9.4413 12.7637 9.32225 13.0061 9.27402C13.2486 9.22579 13.4999 9.25054 13.7284 9.34515C13.9568 9.43976 14.152 9.59998 14.2893 9.80554C14.4267 10.0111 14.5 10.2528 14.5 10.5C14.5 10.8315 14.3683 11.1495 14.1339 11.3839C13.8995 11.6183 13.5815 11.75 13.25 11.75Z" fill="#2D2C2C" fill-opacity="0.4"/>
    </svg>
  `,
  `
    <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path d="M15.1987 4.30447C15.134 4.10228 15.0107 3.92385 14.8445 3.79178C14.6782 3.65971 14.4766 3.57995 14.265 3.56259L12.0881 3.37509L11.2393 1.40072C11.156 1.20749 11.0179 1.0429 10.8421 0.92726C10.6662 0.811622 10.4604 0.75 10.25 0.75C10.0395 0.75 9.8337 0.811622 9.65788 0.92726C9.48206 1.0429 9.34395 1.20749 9.2606 1.40072L8.41247 3.37947L6.23497 3.56259C6.02389 3.57935 5.82251 3.65833 5.6563 3.78954C5.4901 3.92075 5.36654 4.0983 5.30125 4.29973C5.23595 4.50117 5.23186 4.71744 5.28948 4.9212C5.34711 5.12496 5.46386 5.30706 5.62497 5.44447L7.26872 6.84822L6.77685 8.93322C6.72962 9.13849 6.74374 9.35311 6.81745 9.55042C6.89116 9.74774 7.02121 9.91905 7.19145 10.0431C7.36169 10.1671 7.56462 10.2384 7.77502 10.2481C7.98543 10.2578 8.19405 10.2055 8.37497 10.0976L10.25 8.98384L12.125 10.0976C12.3059 10.206 12.5146 10.2587 12.7253 10.2493C12.936 10.2398 13.1392 10.1686 13.3096 10.0444C13.4801 9.92028 13.6102 9.74872 13.6839 9.55113C13.7575 9.35353 13.7714 9.13863 13.7237 8.93322L13.2318 6.84822L14.875 5.44447C15.0356 5.30774 15.1521 5.12646 15.2097 4.92355C15.2673 4.72064 15.2635 4.50521 15.1987 4.30447ZM12.0568 5.87509C11.9063 6.00271 11.7942 6.16962 11.7329 6.35722C11.6717 6.54482 11.6637 6.74574 11.71 6.93759L12.0312 8.29947L10.8012 7.56884C10.6345 7.46943 10.4441 7.41695 10.25 7.41695C10.0559 7.41695 9.86542 7.46943 9.69872 7.56884L8.46872 8.29759L8.78997 6.93759C8.83657 6.74575 8.8288 6.54472 8.76754 6.35705C8.70627 6.16937 8.59392 6.00248 8.4431 5.87509L7.3806 4.96947L8.78935 4.85072C8.98343 4.8349 9.16955 4.76658 9.32776 4.65306C9.48597 4.53955 9.61032 4.38513 9.68747 4.20634L10.25 2.89947L10.8125 4.20572C10.8893 4.38514 11.0136 4.54022 11.1719 4.65431C11.3303 4.76839 11.5167 4.83716 11.7112 4.85322L13.12 4.97197L12.0568 5.87509ZM2.0306 11.5307C1.8897 11.6716 1.69861 11.7508 1.49935 11.7508C1.30009 11.7508 1.10899 11.6716 0.968098 11.5307C0.827202 11.3898 0.748047 11.1987 0.748047 10.9995C0.748047 10.8002 0.827202 10.6091 0.968098 10.4682L4.2181 7.21822C4.35899 7.07732 4.55009 6.99817 4.74935 6.99817C4.94861 6.99817 5.1397 7.07732 5.2806 7.21822C5.42149 7.35911 5.50065 7.55021 5.50065 7.74947C5.50065 7.94872 5.42149 8.13982 5.2806 8.28072L2.0306 11.5307ZM6.2806 11.7807L3.0306 15.0307C2.8897 15.1716 2.69861 15.2508 2.49935 15.2508C2.30009 15.2508 2.10899 15.1716 1.9681 15.0307C1.8272 14.8898 1.74805 14.6987 1.74805 14.4995C1.74805 14.3002 1.8272 14.1091 1.9681 13.9682L5.2181 10.7182C5.35899 10.5773 5.55009 10.4982 5.74935 10.4982C5.94861 10.4982 6.1397 10.5773 6.2806 10.7182C6.42149 10.8591 6.50065 11.0502 6.50065 11.2495C6.50065 11.4487 6.42149 11.6398 6.2806 11.7807ZM10.7806 10.7182C10.8505 10.7879 10.906 10.8707 10.9439 10.9619C10.9817 11.053 11.0012 11.1508 11.0012 11.2495C11.0012 11.3482 10.9817 11.4459 10.9439 11.5371C10.906 11.6282 10.8505 11.711 10.7806 11.7807L7.5306 15.0307C7.3897 15.1716 7.19861 15.2508 6.99935 15.2508C6.80009 15.2508 6.60899 15.1716 6.4681 15.0307C6.3272 14.8898 6.24805 14.6987 6.24805 14.4995C6.24805 14.3002 6.3272 14.1091 6.4681 13.9682L9.7181 10.7182C9.78786 10.6484 9.87072 10.593 9.96193 10.5553C10.0531 10.5175 10.1509 10.4981 10.2496 10.4983C10.3483 10.4984 10.446 10.518 10.5372 10.5559C10.6283 10.5939 10.711 10.6495 10.7806 10.7195V10.7182Z" fill="#2D2C2C" fill-opacity="0.4"/>
    </svg>
  `,
  `
    <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path d="M14.75 6C14.7505 5.93031 14.7412 5.86089 14.7225 5.79375L13.825 2.65687C13.7498 2.39637 13.5922 2.16723 13.3759 2.00375C13.1596 1.84027 12.8961 1.75124 12.625 1.75H3.375C3.10386 1.75124 2.84041 1.84027 2.62409 2.00375C2.40777 2.16723 2.25021 2.39637 2.175 2.65687L1.27875 5.79375C1.25962 5.86083 1.24995 5.93025 1.25 6V7C1.24993 7.40526 1.33956 7.80552 1.51245 8.17205C1.68535 8.53858 1.93722 8.86231 2.25 9.12V13C2.25 13.3315 2.3817 13.6495 2.61612 13.8839C2.85054 14.1183 3.16848 14.25 3.5 14.25H12.5C12.8315 14.25 13.1495 14.1183 13.3839 13.8839C13.6183 13.6495 13.75 13.3315 13.75 13V9.12C14.0628 8.86231 14.3147 8.53858 14.4875 8.17205C14.6604 7.80552 14.7501 7.40526 14.75 7V6ZM13.25 7C13.25 7.33152 13.1183 7.64946 12.8839 7.88388C12.6495 8.1183 12.3315 8.25 12 8.25C11.6685 8.25 11.3505 8.1183 11.1161 7.88388C10.8817 7.64946 10.75 7.33152 10.75 7V6.75H13.25V7ZM2.75 7V6.75H5.25V7C5.25 7.33152 5.1183 7.64946 4.88388 7.88388C4.64946 8.1183 4.33152 8.25 4 8.25C3.66848 8.25 3.35054 8.1183 3.11612 7.88388C2.8817 7.64946 2.75 7.33152 2.75 7ZM6.75 6.75H9.25V7C9.25 7.33152 9.1183 7.64946 8.88388 7.88388C8.64946 8.1183 8.33152 8.25 8 8.25C7.66848 8.25 7.35054 8.1183 7.11612 7.88388C6.8817 7.64946 6.75 7.33152 6.75 7V6.75ZM3.56563 3.25H12.4375L13.0087 5.25H2.99438L3.56563 3.25ZM12.25 12.75H3.75V9.73812C3.8325 9.74563 3.91563 9.75 4 9.75C4.37484 9.75012 4.74572 9.67336 5.08972 9.52448C5.43372 9.37559 5.74353 9.15773 6 8.88437C6.25654 9.15766 6.56636 9.37547 6.91034 9.52436C7.25432 9.67325 7.62517 9.75007 8 9.75007C8.37483 9.75007 8.74568 9.67325 9.08966 9.52436C9.43364 9.37547 9.74346 9.15766 10 8.88437C10.2565 9.15773 10.5663 9.37559 10.9103 9.52448C11.2543 9.67336 11.6252 9.75012 12 9.75C12.0844 9.75 12.1675 9.74563 12.25 9.73812V12.75Z" fill="#2D2C2C" fill-opacity="0.4"/>
    </svg>
  `,
];

let order_sample_avatars = [
  "/assets/images/avatar-1.png",
  "/assets/images/avatar-2.png",
  "/assets/images/avatar-3.png",
  "/assets/images/avatar-4.png",
];

let businessProfile = {
  id: "543219876",
  ownersName: "Jane Doe",
  businessName: "Jane's restaurant",
  businessAddress: "123 Main St, Anytown, USA",
  businessPhone: "123-456-7890",
  businessWhatsappNo: "123-456-7890",
  businessAvatar: "./assets/icons/avatars/avatar-2.svg",
  businessCover:
    "https://res.cloudinary.com/dddotdmjo/image/upload/v1735999712/food-items/food-36_ljl7y5.jpg",
  businessHours: {
    morning: {
      openingHours: "12:00",
      closingHours: "3:00",
    },
    evening: {
      openingHours: "12:00",
      closingHours: "3:00",
    },
  },
  businessType: "Kitchen",
  currency: "dollars",
  otherSettings: {
    restaurant: {
      vegetarian: {
        delivery: {
          enabled: true,
          minPrice: 5.0,
          deliveryTime: 30,
          distance: 30,
        },
        dinein: {
          enabled: true,
          minPrice: 20.0,
          dineinTables: 30,
        },
        takeaway: {
          enabled: true,
          minPrice: 10.0,
        },
        openingHours: {
          morning: {
            openingHours: "12:00",
            closingHours: "3:00",
          },
          evening: {
            openingHours: "12:00",
            closingHours: "3:00",
          },
        },
        tax: {
          enabled: true,
          taxRate: 0.1,
          status: "inclusive",
          platformFee: 50.0,
        },
      },
      nonvegetarian: {
        delivery: {
          enabled: true,
          minPrice: 5.0,
          deliveryTime: 30,
          distance: 30,
        },
        dinein: {
          enabled: true,
          minPrice: 20.0,
          dineinTables: 30,
        },
        takeaway: {
          enabled: true,
          minPrice: 10.0,
        },
        openingHours: {
          morning: {
            openingHours: "12:00",
            closingHours: "3:00",
          },
          evening: {
            openingHours: "12:00",
            closingHours: "3:00",
          },
        },
        tax: {
          enabled: true,
          taxRate: 0.1,
          status: "inclusive",
          platformFee: 50.0,
        },
      },
    },
    cuisines: ["austrian", "russian", "peruvian"],
    paymentOption: "online",
  },
  businessStatus: "open",
};

const orders = [
  {
    id: "ZP37646739299399",
    type: "delivery",
    date: "2 Minutes Ago",
    location: "2972 Westheimer Rd. Santa Ana, Illinois 85486",
    customer: {
      name: "Brooklyn Simmons",
      avatar: "./assets/icons/avatars/avatar-1.svg",
      phone: "+1 837399989",
      id: "ZP37646739299399",
    },
    tableNumber: "07",
    items: [
      { name: "Burger", price: "14.99" },
      { name: "Fries", price: "4.99" },
    ],
    total: "$19.98",
    partner: {
      id: "ZP37646739299399",
      name: "Alice Carey",
      avatar: "./assets/icons/avatars/avatar-1.svg",
      phone: "123-456-7890",
      rating: "3",
    },
    specialRequest: {
      status: true,
      selectedOptions: ["contactless", "cutlery"],
    },
    customerNote:
      "Please leave the package at the front door if no one answers.",
    businessID: "543219876",
  },
  {
    id: "ZP37646738299399",
    type: "dine-in",
    date: "2 Minutes Ago",
    location: "2972 Westheimer Rd. Santa Ana, Illinois 85486",
    customer: {
      name: "Jane Doe",
      avatar: "./assets/icons/avatars/avatar-1.svg",
      phone: "+1 837399989",
      id: "ZP37646739299399",
    },
    tableNumber: "07",
    items: [
      { name: "Burger", price: "14.99" },
      { name: "Fries", price: "4.99" },
    ],
    total: "$19.98",
    partner: {
      id: "ZP37646739299399",
      name: "Alice Carey",
      avatar: "./assets/icons/avatars/avatar-1.svg",
      phone: "123-456-7890",
      rating: "3",
    },
    specialRequest: {
      status: true,
      selectedOptions: ["contactless"],
    },
    customerNote: "",
    businessID: "543219876",
  },
];

let dummyData = null;

async function loadData() {
  try {
    const response = await fetch("./assets/data.json");
    dummyData = await response.json();
    return dummyData;
  } catch (error) {
    console.error("Error loading data:", error);
    throw error;
  }
}

// Basic CRUD operations
async function get(collection, id = null) {
  if (id === null) {
    return dummyData[collection];
  }
  return dummyData[collection].find((item) => item.id === id);
}

async function add(collection, item) {
  const newId = Math.max(...dummyData[collection].map((item) => item.id)) + 1;
  const newItem = { ...item, id: newId };
  dummyData[collection].push(newItem);
  return newItem;
}

async function update(collection, id, updates) {
  const index = dummyData[collection].findIndex((item) => item.id === id);
  if (index !== -1) {
    dummyData[collection][index] = {
      ...dummyData[collection][index],
      ...updates,
    };
    return dummyData[collection][index];
  }
  return null;
}

async function remove(collection, id) {
  const index = dummyData[collection].findIndex((item) => item.id === id);

  if (index !== -1) {
    const deletedItem = dummyData[collection].splice(index, 1)[0];

    return deletedItem;
  }

  return null;
}

async function updateIfHasValue(collection, id, key, value, updates) {
  const index = dummyData[collection].findIndex((item) => item.id === id);

  if (index !== -1 && dummyData[collection][index][key] === value) {
    dummyData[collection][index] = {
      ...dummyData[collection][index],
      ...updates,
    };
    return dummyData[collection][index]; // Return the updated object
  }

  return null; // Return null if object is not found or condition isn't met
}

async function remove(collection, id) {
  const index = dummyData[collection].findIndex((item) => item.id === id);
  if (index !== -1) {
    return dummyData[collection].splice(index, 1)[0];
  }
  return null;
}

async function getFiltered(collection, filters = {}) {
  if (
    !filters ||
    typeof filters !== "object" ||
    Object.keys(filters).length === 0
  ) {
    throw new Error("At least one filter must be provided.");
  }

  return dummyData[collection].filter((item) =>
    Object.entries(filters).every(([key, value]) => item[key] === value)
  );
}

const populateDeliveryPartnerModal = async (partnerId) => {
  const deliveryPartnerModal = document.getElementById("partner-profile");
  const partner = await get("delivery-partners", partnerId);

  // Populate partner details
  deliveryPartnerModal.querySelector(".name").innerText = partner.name;
  deliveryPartnerModal.querySelector(".distance").innerText = partner.distance;
  deliveryPartnerModal.querySelector(".phone").innerText = partner.phone;
  deliveryPartnerModal.querySelector(".email").innerText = partner.email;
  deliveryPartnerModal.querySelector(".avatar").src = partner.avatar;
  deliveryPartnerModal.querySelector(
    ".total-earnings .number"
  ).innerText = `$${partner.totalEarnings}`;
  deliveryPartnerModal.querySelector(".completed-orders .number").innerText =
    partner.completedOrders;
  deliveryPartnerModal.querySelector(".average-deliveries .number").innerText =
    partner.averageDeliveries;
  deliveryPartnerModal.querySelector(".last-record > span").innerText =
    formatISODate(partner.lastDelivery);

  // Calculate average rating
  const averageRating =
    partner.reviews.reviewsList.length > 0
      ? partner.reviews.reviewsList.reduce(
          (sum, review) => sum + review.rating,
          0
        ) / partner.reviews.reviewsList.length
      : 0;

  deliveryPartnerModal.querySelector(".reviews-section .title").innerText = `${
    partner.name.split(" ")[0]
  }'s Reviews`;
  deliveryPartnerModal.querySelector(".reviews-container .number").innerText =
    averageRating.toFixed(2);
  deliveryPartnerModal.querySelector(
    ".reviews-container .title"
  ).innerText = `${partner.reviews.totalReviews} Reviews`;

  // Populate reviews
  const reviewsList = deliveryPartnerModal.querySelector(
    ".column-two .reviews-section .reviews"
  );
  reviewsList.innerHTML = "";
  partner.reviews.reviewsList.forEach((item) => {
    const dateCreated = formatISODate(item.dateCreated);
    const listItem = document.createElement("div");
    listItem.classList.add("review-card");
    listItem.innerHTML = `
          <div class="review">
              <img src="${item.avatar}" alt="avatar" width="30" height="30" />
              <div>
                  <span class="title">${item.name}</span>
                  <span class="id">${item.id}</span>
                  <p class="text">${item.reviewText}</p>
                  <div class="other-info">
                      <span class="date">${dateCreated}</span>
                      <p class="rating">
                          <img src="./assets/icons/star.svg" class="star" width="15" />
                          <span>${item.rating}</span>
                      </p>
                  </div>
              </div>
          </div>
      `;
    reviewsList.appendChild(listItem);
  });

  deliveryPartnerModal.showModal();

  // Open Block Modal
  const blockModal = document.getElementById("block-modal");
  const blockButton = deliveryPartnerModal.querySelector(
    "#partner-block-button"
  );
  const response = await get("restaurants", restaurantID);
  const restaurantData = response;

  let isBlocked = restaurantData.blocked.blockedPartners.some(
    (item) => item.id == partner.id
  );
  updateBlockButton(blockButton, isBlocked);

  blockButton.addEventListener("click", () => {
    toggleModalContent(
      isBlocked,
      blockModal,
      blockButton,
      partner,
      restaurantData
    );
    blockModal.showModal();
  });

  const closeButton = deliveryPartnerModal.querySelector(".close-button");
  closeButton.addEventListener("click", () => {
    deliveryPartnerModal.close();
  });
};

const updateBlockButton = (blockButton, isBlocked) => {
  if (isBlocked) {
    blockButton.classList.remove("block");
    blockButton.classList.add("unblock");
    blockButton.innerHTML = `
          <span> Unblock Delivery Partner </span>
          <img src="./assets/icons/double-check.svg" alt="check-icon"/>
      `;
  } else {
    blockButton.classList.add("block");
    blockButton.classList.remove("unblock");
    blockButton.innerHTML = `
          <span> Block Delivery Partner </span>
          <img src="./assets/icons/block-icon.svg" alt="block-icon"/>
      `;
  }
};

const toggleModalContent = (
  isBlocked,
  blockModal,
  blockButton,
  partner,
  restaurantData
) => {
  blockModal.style.zIndex = 50;
  blockModal.innerHTML = isBlocked
    ? `
      <div class="modal-heading">
          <p>
              <img src="./assets/icons/smiles.svg" alt="smile" />
              <span>Oh Yeah! 👍🏾</span>
          </p>
          <button type="button" class="close-button">
              <img src="./assets/icons/close.svg" alt="close icon" class="close-icon" />
          </button>
      </div>
      <div class="modal-body">
          <h2>Great! <br/> ${partner.name} will be able to run deliveries for your restaurant, when you unblock them.</h2>
          <p class="button-section">
              <button id="unblock-confirm-button" class="open-modal-btn">
                  <span> Unblock Delivery Partner </span>
                  <img src="./assets/icons/double-check.svg" alt="check-icon"/>
              </button>
              <button id="cancel-button">Cancel</button>
          </p>
      </div>
  `
    : `
      <div class="modal-heading">
          <p>
              <img src="./assets/icons/frown.svg" alt="frown" />
              <span>Oh No!</span>
          </p>
          <button type="button" class="close-button">
              <img src="./assets/icons/close.svg" alt="close icon" class="close-icon" />
          </button>
      </div>
      <div class="modal-body">
          <h2>Do you want to block this Delivery Partner?</h2>
          <p class="button-section">
              <button id="block-confirm-button" class="open-modal-btn">
                  <span> Block Delivery Partner </span>
                  <img src="./assets/icons/block-icon.svg" alt="block-icon"/>
              </button>
              <button id="cancel-button">Cancel</button>
          </p>
      </div>
  `;

  // Add event listeners to new buttons
  document
    .getElementById("block-confirm-button")
    ?.addEventListener("click", async () => {
      blockModal.close();
      try {
        let partnerDetails = { id: partner.id, name: partner.name };
        restaurantData.blocked.blockedPartners.push(partnerDetails);
        await update("restaurants", restaurantID, {
          blocked: restaurantData.blocked,
        });
      } catch (error) {
        createAlert(
          "Couldn't block delivery partner",
          "./assets/icons/bin.svg",
          "red-5"
        );
      } finally {
        updateBlockButton(blockButton, true);
      }
    });

  document
    .getElementById("unblock-confirm-button")
    ?.addEventListener("click", async () => {
      blockModal.close();
      try {
        restaurantData.blocked.blockedPartners =
          restaurantData.blocked.blockedPartners.filter(
            (partner) => partner.id !== partner.id
          );
        await update("restaurants", restaurantID, {
          blocked: restaurantData.blocked,
        });
      } catch (error) {
        createAlert(
          "Couldn't unblock delivery partner",
          "./assets/icons/bin.svg",
          "red-5"
        );
      } finally {
        updateBlockButton(blockButton, false);
      }
    });

  document
    .querySelectorAll(".close-button, #cancel-button")
    .forEach((button) =>
      button.addEventListener("click", () => blockModal.close())
    );
};

const createReviewCard = (item, customer, isRestaurantReview) => {
  const listItem = document.createElement("div");
  listItem.classList.add("review-card");
  const isCommented = Object.keys(item.comment).length !== 0;

  if (isRestaurantReview) {
    listItem.classList.add("restaurant-review");
  }
  if (isRestaurantReview && isCommented) {
    listItem.classList.add("commented");
  }

  listItem.innerHTML = `
        <div class="review">
            <img src="${customer.avatar}" alt="avatar" width="30" height="30" />
            <div>
                <p class="title">
                    <span class="title">${customer.name}</span>
                    <button>
                        <svg width="13" height="13" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" class="arrow">
                          <path
                            d="M13.5306 6.53061L8.5306 11.5306C8.46092 11.6005 8.37813 11.656 8.28696 11.6939C8.1958 11.7317 8.09806 11.7512 7.99935 11.7512C7.90064 11.7512 7.8029 11.7317 7.71173 11.6939C7.62057 11.656 7.53778 11.6005 7.4681 11.5306L2.4681 6.53061C2.3272 6.38972 2.24805 6.19862 2.24805 5.99936C2.24805 5.80011 2.3272 5.60901 2.4681 5.46811C2.60899 5.32722 2.80009 5.24806 2.99935 5.24806C3.19861 5.24806 3.3897 5.32722 3.5306 5.46811L7.99997 9.93749L12.4693 5.46749C12.6102 5.32659 12.8013 5.24744 13.0006 5.24744C13.1999 5.24744 13.391 5.32659 13.5318 5.46749C13.6727 5.60838 13.7519 5.79948 13.7519 5.99874C13.7519 6.198 13.6727 6.38909 13.5318 6.52999L13.5306 6.53061Z"
                            fill="#2D2C2C"
                          />
                        </svg>
                    </button>
                </p>
                <span class="id">${customer.id}</span>
                <p class="text">${item.reviewText}</p>
                <div class="other-info">
                    <span class="date">${formatISODate(item.dateCreated)}</span>
                    <p class="rating">
                        <img src="./assets/icons/chat-bubble.svg" class="chat-bubble" width="15" style="margin-right: 10px" />
                        <img src="./assets/icons/star.svg" class="star" width="15" />
                        <span>${item.rating}</span>
                    </p>
                </div>
                <p class="reply">
                    <button class="reply-button">
                        <span>Reply</span>
                        <img src="./assets/icons/curved-arrow.svg" class="arrow" width="15" />
                    </button>
                </p>
            </div>
        </div>
        <div class="comment-box">
            <img src="${
              restaurantData.businessAvatar
            }" alt="avatar" width="26" height="26" />
            <div class="comment-content">
                <div class="comment">
                    <div class="comment-input">
                        <input id="new-comment" type="text" placeholder="Write a reply" />
                        <button class="send-new-comment-button">
                            <svg width="13" height="13" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg">
                              <path
                                d="M14.3838 0.616251C14.2267 0.459113 14.0305 0.346686 13.8156 0.290551C13.6006 0.234415 13.3745 0.236598 13.1606 0.296876L13.1469 0.301251L1.15252 3.9375C0.908896 4.00814 0.692525 4.15122 0.53215 4.34775C0.371775 4.54428 0.274986 4.78495 0.254639 5.03779C0.234292 5.29063 0.291351 5.54368 0.418236 5.76333C0.54512 5.98297 0.735826 6.15882 0.965022 6.2675L6.23502 8.7675L8.73502 14.0375C8.83481 14.2516 8.99375 14.4326 9.19309 14.5593C9.39242 14.686 9.62384 14.753 9.86002 14.7525C9.89565 14.7525 9.9319 14.7525 9.96815 14.7481C10.2208 14.728 10.4612 14.6308 10.6569 14.4698C10.8526 14.3087 10.9941 14.0915 11.0625 13.8475L14.6988 1.85313C14.7006 1.84867 14.7021 1.84407 14.7031 1.83938C14.7634 1.62551 14.7656 1.39945 14.7095 1.18447C14.6533 0.96948 14.5409 0.773342 14.3838 0.616251ZM9.81252 12.8075L7.70502 8.35813L10.5331 5.53C10.674 5.38911 10.7532 5.19801 10.7532 4.99875C10.7532 4.79949 10.674 4.6084 10.5331 4.4675C10.3923 4.32661 10.2012 4.24745 10.0019 4.24745C9.80264 4.24745 9.61154 4.32661 9.47065 4.4675L6.64252 7.29563L2.19252 5.1875L13.125 1.875L9.81252 12.8075Z"
                                fill="white"
                              />
                            </svg>
                        </button>
                    </div>
                    <button type="button" class="close-button">
                        <img src="./assets/icons/close.svg" alt="close icon" class="close-icon" />
                    </button>
                </div>
                <div class="comment">
              <div>
                <p class="title">
                  <span>${restaurantData.ownersName}</span>
                </p>
                <p class="text">${
                  isCommented ? item.comment.commentText : ""
                }</p>
                <div class="comment-input">
                  <textarea
                    type="text"
                    id="old-comment"
                    name="old-comment"
                  >${isCommented ? item.comment.commentText : ""}</textarea>
                  <button class="send-old-comment-button">
                    <svg
                      width="13"
                      height="13"
                      viewBox="0 0 15 15"
                      fill="none"
                      xmlns="http://www.w3.org/2000/svg"
                    >
                      <path
                        d="M14.3838 0.616251C14.2267 0.459113 14.0305 0.346686 13.8156 0.290551C13.6006 0.234415 13.3745 0.236598 13.1606 0.296876L13.1469 0.301251L1.15252 3.9375C0.908896 4.00814 0.692525 4.15122 0.53215 4.34775C0.371775 4.54428 0.274986 4.78495 0.254639 5.03779C0.234292 5.29063 0.291351 5.54368 0.418236 5.76333C0.54512 5.98297 0.735826 6.15882 0.965022 6.2675L6.23502 8.7675L8.73502 14.0375C8.83481 14.2516 8.99375 14.4326 9.19309 14.5593C9.39242 14.686 9.62384 14.753 9.86002 14.7525C9.89565 14.7525 9.9319 14.7525 9.96815 14.7481C10.2208 14.728 10.4612 14.6308 10.6569 14.4698C10.8526 14.3087 10.9941 14.0915 11.0625 13.8475L14.6988 1.85313C14.7006 1.84867 14.7021 1.84407 14.7031 1.83938C14.7634 1.62551 14.7656 1.39945 14.7095 1.18447C14.6533 0.96948 14.5409 0.773342 14.3838 0.616251ZM9.81252 12.8075L7.70502 8.35813L10.5331 5.53C10.674 5.38911 10.7532 5.19801 10.7532 4.99875C10.7532 4.79949 10.674 4.6084 10.5331 4.4675C10.3923 4.32661 10.2012 4.24745 10.0019 4.24745C9.80264 4.24745 9.61154 4.32661 9.47065 4.4675L6.64252 7.29563L2.19252 5.1875L13.125 1.875L9.81252 12.8075Z"
                        fill="white"
                      />
                    </svg>
                  </button>
                </div>
                <div class="other-info">
                  <span class="date">${
                    isCommented ? formatISODate(item.comment.dateCreated) : ""
                  }</span>
                  <p class="edit">
                    <span> Edit</span>
                    <svg
                      width="15"
                      height="15"
                      viewBox="0 0 32 32"
                      fill="none"
                      xmlns="http://www.w3.org/2000/svg"
                    >
                      <path
                        d="M28.7675 8.81762L23.1825 3.23137C22.9504 2.99912 22.6747 2.81488 22.3713 2.68918C22.0679 2.56348 21.7428 2.49878 21.4144 2.49878C21.086 2.49878 20.7608 2.56348 20.4574 2.68918C20.1541 2.81488 19.8784 2.99912 19.6463 3.23137L4.23251 18.6464C3.99951 18.8778 3.81479 19.1533 3.68905 19.4567C3.56331 19.7601 3.49905 20.0854 3.50001 20.4139V26.0001C3.50001 26.6632 3.7634 27.299 4.23224 27.7679C4.70108 28.2367 5.33697 28.5001 6.00001 28.5001H11.5863C11.9147 28.501 12.24 28.4367 12.5434 28.311C12.8468 28.1853 13.1222 28.0006 13.3538 27.7676L28.7675 12.3526C29.2362 11.8838 29.4995 11.248 29.4995 10.5851C29.4995 9.92221 29.2362 9.28644 28.7675 8.81762ZM11.375 25.5001H6.50001V20.6251L17 10.1251L21.875 15.0001L11.375 25.5001ZM24 12.8751L19.125 8.00012L21.4175 5.70762L26.2925 10.5826L24 12.8751Z"
                        fill="#ffb016"
                      />
                    </svg>
                  </p>
                  <button class="cancel-button">Cancel</button>
                </div>
              </div>
            </div>
            </div>
        </div>
    `;

  // Setup interactions
  const commentBoxToggles = listItem.querySelectorAll(
    ".review p.title button, .review .reply button"
  );
  commentBoxToggles.forEach((toggle) => {
    toggle.addEventListener("click", () => {
      listItem.classList.toggle("open");
    });
  });

  const commentBox = listItem.querySelector(".comment-box");
  const editButton = listItem.querySelector(".comment-box .edit");
  editButton.addEventListener("click", () => {
    commentBox.classList.add("edit");
  });

  const cancelButton = listItem.querySelector(".comment-box .cancel-button");
  cancelButton.addEventListener("click", () => {
    commentBox.classList.remove("edit");
  });

  const sendButton = listItem.querySelector(
    ".comment-input .send-old-comment-button"
  );
  sendButton.addEventListener("click", async () => {
    let updatedData = {
      comment: {
        id: "3333",
        commentText: commentBox
          .querySelector(".comment-input textarea")
          .value.trim(),
        dateCreated: new Date(),
      },
    };
    try {
      sendButton.disabled = true;
      const updatedReview = await update("reviews", item.id, updatedData);
      if (updatedReview) {
        createAlert("Reply sent", "./assets/icons/bin.svg", "green-4");
        populateReviewsPage();
      }
    } catch (error) {
      createAlert("couldn't update reply", "./assets/icons/bin.svg", "red-5");
    } finally {
      commentBox.classList.remove("edit");
      sendButton.disabled = false;
    }
  });

  const sendNewButton = listItem.querySelector(
    ".comment-input .send-new-comment-button"
  );
  sendNewButton.addEventListener("click", async () => {
    let updatedData = {
      comment: {
        id: "3333",
        commentText: commentBox
          .querySelector(".comment-input #new-comment")
          .value.trim(),
        dateCreated: new Date(),
      },
    };
    try {
      sendButton.disabled = true;
      const updatedReview = await update("reviews", item.id, updatedData);
      if (updatedReview) {
        createAlert("Reply sent", "./assets/icons/bin.svg", "green-6");
        populateReviewsPage();
      }
    } catch (error) {
      createAlert("couldn't send reply", "./assets/icons/bin.svg", "red-5");
    } finally {
      commentBox.classList.remove("edit");
      sendButton.disabled = false;
    }
  });

  const closeButton = listItem.querySelector(".comment .close-button");
  closeButton.addEventListener("click", () => {
    listItem.classList.remove("open");
  });

  return listItem;
};

// Function to populate the modal
const populateCustomerProfileModal = async (customerId) => {
  const customerProfileModal = document.getElementById("customer-profile");

  const customer = await get("customers", customerId);
  const customerReviews = await getFiltered("reviews", {
    customerID: customerId,
  });
  console.log(customerReviews);

  const { restaurantReviews, otherReviews } = customerReviews.reduce(
    (acc, obj) => {
      obj.businessID === restaurantID
        ? acc.restaurantReviews.push(obj)
        : acc.otherReviews.push(obj);
      return acc;
    },
    { restaurantReviews: [], otherReviews: [] }
  );

  const calculateAverageRating = (reviews) => {
    if (reviews.length === 0) return 0;

    const totalRating = reviews.reduce((sum, review) => sum + review.rating, 0);
    return (totalRating / reviews.length).toFixed(1);
  };

  // Populate customer details
  customerProfileModal.querySelector(".name").innerText = customer.name;
  customerProfileModal.querySelector(".phone").innerText = customer.phone;
  customerProfileModal.querySelector(".email").innerText = customer.email;
  customerProfileModal.querySelector(".avatar").src = customer.avatar;
  customerProfileModal.querySelector(
    ".total-spending .number"
  ).innerText = `$${customer.totalSpending}`;
  customerProfileModal.querySelector(
    ".average-spending .number"
  ).innerText = `$${customer.averageSpending}`;
  customerProfileModal.querySelector(".total-orders .number").innerText =
    customer.totalOrders;
  customerProfileModal.querySelector(".last-record > span").innerText =
    formatISODate(customer.lastPurchase);

  // Populate reviews
  const reviewsContainer =
    customerProfileModal.querySelector(".reviews-container");
  reviewsContainer.querySelector(".number").innerText =
    calculateAverageRating(customerReviews);
  reviewsContainer.querySelector(
    ".title"
  ).innerText = `${customerReviews.length} Reviews`;

  // Populate restaurant reviews
  const restaurantReviewsList = customerProfileModal.querySelector(
    ".column-two .reviews-section .reviews"
  );
  restaurantReviewsList.innerHTML = "";
  restaurantReviews.forEach((item) => {
    const listItem = createReviewCard(item, customer, true);
    restaurantReviewsList.appendChild(listItem);
  });

  // Populate other reviews
  const otherReviewsList = customerProfileModal.querySelector(
    ".column-three .reviews-section .reviews"
  );
  otherReviewsList.innerHTML = "";
  otherReviews.forEach((item) => {
    const listItem = createReviewCard(item, customer, false);
    otherReviewsList.appendChild(listItem);
  });

  // Setup block/unblock button
  const blockModal = document.getElementById("block-modal");
  const blockButton = customerProfileModal.querySelector("#block-button");
  blockButton.classList.toggle("block", !customer.blockStatus);
  blockButton.classList.toggle("unblock", customer.blockStatus);
  blockButton.innerHTML = `
      <span>${customer.blockStatus ? "Unblock" : "Block"} Customer</span>
      <img src="./assets/icons/${
        customer.blockStatus ? "double-check" : "block-icon"
      }.svg" alt="icon"/>
  `;

  blockButton.addEventListener("click", () => {
    const isBlock = blockButton.classList.contains("block");
    toggleBlockModalContent(isBlock, blockModal, blockButton, customer);
    blockModal.showModal();
  });

  // Setup close button for the modal
  const closeButton = customerProfileModal.querySelector(".close-button");
  closeButton.addEventListener("click", () => {
    customerProfileModal.close();
  });

  // Function to toggle block modal content
  const toggleBlockModalContent = (
    isBlock,
    blockModal,
    blockButton,
    customer
  ) => {
    blockModal.style.zIndex = 50;
    blockModal.classList.toggle("block", isBlock);
    blockModal.classList.toggle("unblock", !isBlock);
    blockModal.innerHTML = `
          <div class="modal-heading">
              <p>
                  <img src="./assets/icons/${
                    isBlock ? "frown" : "smiles"
                  }.svg" alt="${isBlock ? "frown" : "smile"}" />
                  <span>${isBlock ? "Oh No!" : "Oh Yeah! 👍🏾"}</span>
              </p>
              <button type="button" class="close-button">
                  <img src="./assets/icons/close.svg" alt="close icon" class="close-icon" />
              </button>
          </div>
          <div class="modal-body">
              <h2>${
                isBlock
                  ? "Do you want to block this customer?"
                  : "Great! They will be able to order from your restaurant when you unblock them."
              }</h2>
              <p class="button-section">
                  <button id="${
                    isBlock ? "block-confirm-button" : "unblock-confirm-button"
                  }" class="open-modal-btn">
                      <span>${isBlock ? "Block" : "Unblock"} Customer</span>
                      <img src="./assets/icons/${
                        isBlock ? "block-icon" : "double-check"
                      }.svg" alt="icon"/>
                  </button>
                  <button id="cancel-button">Cancel</button>
              </p>
          </div>
      `;

    // Add event listeners to new buttons
    document
      .getElementById(
        `${isBlock ? "block-confirm-button" : "unblock-confirm-button"}`
      )
      .addEventListener("click", () => {
        blockModal.close();
        customer.blockStatus = isBlock;
        toggleBlockModalContent(!isBlock, blockModal, blockButton, customer);
      });
    document
      .querySelectorAll(".close-button, #cancel-button")
      .forEach((button) =>
        button.addEventListener("click", () => blockModal.close())
      );
  };
  customerProfileModal.showModal();
};

const populateOrderModal = async (orderId, type) => {
  const orderModal = document.getElementById("order-details");

  const order = await get("orders", orderId);

  if (!order) {
    console.error("Order not found!");
    return;
  }

  console.log(order);

  // Update modal content dynamically
  orderModal.querySelector(".order-type").textContent = order.type;
  orderModal.querySelector(".order-id").textContent = order.id;
  orderModal.querySelector(".order-date").textContent = order.date;
  orderModal.querySelector(".location").textContent = order.location;
  orderModal.querySelector(".order-name").textContent = order.customer.name;
  orderModal.querySelector(".order-user .user.picture").src =
    order.customer.avatar;
  orderModal.querySelector(".order-phone .phone").textContent =
    order.customer.phone;
  orderModal.querySelector(".table-number").textContent = order.tableNumber;

  orderModal.querySelector(".partner-name").textContent = order.partner.name;
  orderModal.querySelector(".partner-details .user.picture").src =
    order.partner.avatar;
  orderModal.querySelector("footer .phone span").textContent =
    order.partner.phone;
  orderModal.querySelector("footer .rating span").textContent =
    order.partner.rating;

  // Populate order items
  const itemsList = orderModal.querySelector(".order");
  itemsList.innerHTML = "";
  order.items.forEach((item) => {
    const listItem = document.createElement("li");
    listItem.classList.add("item");
    listItem.innerHTML = `
      <span class="name">${item.name}</span>
      <span class="price">${item.price}</span>
    `;
    itemsList.appendChild(listItem);
  });

  // Add total
  const totalItem = document.createElement("li");
  totalItem.classList.add("total");
  totalItem.innerHTML = `
    <span class="name">Total</span>
    <span class="price">${order.total}</span>
  `;
  itemsList.appendChild(totalItem);

  if (order.specialRequest.status) {
    order.specialRequest.selectedOptions.forEach((option) => {
      const checkbox = orderModal.querySelector(
        `input[type="checkbox"][value="${option}"]`
      );
      if (checkbox) {
        checkbox.checked = true;
      }
    });
  }

  orderModal.querySelector("#note").value = order.customerNote;
  orderModal.showModal();
  if (type && type === "view") {
    orderModal.classList.add("view");
  }

  const orderCloseButton = orderModal.querySelector(".close-button");
  orderCloseButton.addEventListener("click", () => {
    orderModal.close();
    if (orderModal.classList.contains("view")) {
      orderModal.classList.remove("view");
    }
  });

  orderModal.querySelector(".order-name").addEventListener("click", () => {
    populateCustomerProfileModal(order.customer.id);
  });
  orderModal.querySelector(".partner-name").addEventListener("click", () => {
    populateDeliveryPartnerModal(order.partner.id);
  });
};

const populateNotifications = async () => {
  const restaurantNotifications = await getFiltered("notifications", {
    businessID: restaurantID,
  });

  const container = document.querySelector(".top-nav .notification-items");
  const template = document.querySelector(".top-nav #notfication-template");

  function populateContainer() {
    container
      .querySelectorAll(":not(template)")
      .forEach((child) => child.remove());

    restaurantNotifications.forEach(async (item) => {
      const clone = template.content.cloneNode(true);
      const titleElement = clone.querySelector(".notification-title");
      const priceElement = clone.querySelector(".price");
      const timeElement = clone.querySelector(".time");

      const itemData = await getFiltered("orders", {
        id: item.id,
      });

      titleElement.textContent = item.title;
      priceElement.textContent = `$${itemData[0].total}`;
      timeElement.textContent = formatISODate(item.dateCreated);

      clone.id = item.id;

      clone.firstElementChild.addEventListener("click",  () => {
        console.log(item.orderId)
        populateOrderModal(item.orderId)
      });
      container.appendChild(clone);
    });
  }
  populateContainer();

  const notificationCounter = notification_button.querySelector(
    ".notification-counter"
  );
  if (restaurantNotifications.length > 0) {
    notificationCounter.textContent = restaurantNotifications.length;
  } else {
    notificationCounter.style.display = "none";
  }
};

// Populate Earnings Section
const populateEarnings = async () => {
  const restaurantOrders = await getFiltered("orders", {
    businessID: restaurantID,
  });
  const container = document.querySelector(".earnings-list ul");
  const template = document.querySelector("#earning-template");

  let orderTotal = 0;

  function populateContainer() {
    container
      .querySelectorAll(":not(template)")
      .forEach((child) => child.remove());

    restaurantOrders.forEach(async (item) => {
      const clone = template.content.cloneNode(true);
      const idElement = clone.querySelector(".id");
      const priceElement = clone.querySelector(".amount");
      const timeElement = clone.querySelector(".date");

      idElement.textContent = item.id;
      priceElement.textContent = `$${item.total}`;
      timeElement.textContent = formatISODate(item.dateCreated);

      orderTotal += item.total;

      clone.id = item.id;
      container.appendChild(clone);
    });

    const earningTotalElement1 = document.querySelector(
      "#home .earnings-container .earnings-total span, #dashboard-header .navigation-container1 > div.nav-items > div > p > span"
    );
    earningTotalElement1.textContent = orderTotal;
  }
  populateContainer();
};

// Count dishes
function countAllDishes(data) {
  const categorizedCount = data.categories.reduce(
    (total, category) => total + (category.dishes ? category.dishes.length : 0),
    0
  );
  const uncategorizedCount = data.uncategorizedItems.length;
  return categorizedCount + uncategorizedCount;
}
// Populate Information Cards
const populateInfoCards = async () => {
  const restaurantOrders = await getFiltered("orders", {
    businessID: restaurantID,
  });
  let deliveryOrders = restaurantOrders.filter((item) =>
    Object.entries({ type: "delivery" }).every(
      ([key, value]) => item[key] === value
    )
  );
  let ordersInTransit = deliveryOrders.filter((item) =>
    Object.entries({ orderStatus: "transit" }).every(
      ([key, value]) => item[key] === value
    )
  );

  const ordersCard = document.querySelector("#home .cards .card:nth-child(1)");
  const orderTotalsElement = ordersCard.querySelector("#orders-total");
  orderTotalsElement.textContent = restaurantOrders.length;

  const dineinCard = document.querySelector("#home .cards .card:nth-child(2)");
  const dineinTotalElement = dineinCard.querySelector("#dinein-total");
  dineinTotalElement.textContent =
    restaurantOrders.length - deliveryOrders.length;

  const deliveryCard = document.querySelector(
    "#home .cards .card:nth-child(3)"
  );
  const deliveryTotalElement = deliveryCard.querySelector("#delivery-total");
  const transitDeliveryTotalElement = deliveryCard.querySelector(
    "#transit-delivery-total"
  );
  deliveryTotalElement.textContent = deliveryOrders.length;
  transitDeliveryTotalElement.textContent = ordersInTransit.length;

  const restaurantMenus = await getFiltered("menus", {
    businessID: restaurantID,
  });
  const dishesCard = document.querySelector("#home .cards .card:nth-child(4)");
  const menuTotalElement = dishesCard.querySelector("#menu-total");
  const dishesTotalElement = dishesCard.querySelector("#dishes-total");

  dishesTotalElement.textContent = countAllDishes(restaurantMenus[0]);
  menuTotalElement.textContent = restaurantMenus[0].categories.length;
};

const populateRecentOrder = (data) => {
  const container = document.querySelector(
    "#home .containers .recent-orders .orders"
  );
  const deliveryOrderTemplate = document.querySelector("#order-template");

  function populateContainer() {
    container
      .querySelectorAll(":not(template)")
      .forEach((child) => child.remove());

    const clone = deliveryOrderTemplate.content.cloneNode(true);
    const idElement = clone.querySelector(".order-id");
    const orderTypeElement = clone.querySelector(".order-type");
    const customerAvatarElement = clone.querySelector(".customer-profile");
    const customerNameElement = clone.querySelector(".order-name");
    const totalOrderElement = clone.querySelector(".order .total .price");
    const timeCreatedElement = clone.querySelector(".order-time");

    idElement.textContent = data.id;
    orderTypeElement.textContent = data.type;
    customerAvatarElement.src = data.customer.avatar;
    customerNameElement.textContent = data.customer.name;
    totalOrderElement.textContent = `$${data.total}`;
    timeCreatedElement.textContent = formatISODate(data.dateCreated);

    const orderContainer = clone.querySelector(".delivery-info .order");
    const totalElement = clone.querySelector(".order .total");

    for (let i = 0; i < data.items.length; i++) {
      const orderItemElement = document.createElement("li");
      orderItemElement.id = `order-item-${i + 1}`;
      orderItemElement.classList.add("item");

      const itemName = document.createElement("span");
      itemName.classList.add("name");

      const itemPrice = document.createElement("span");
      itemPrice.classList.add("price");

      orderItemElement.append(itemName, itemPrice);

      itemName.textContent = data.items[i].name;
      itemPrice.textContent = `$${data.items[i].price}`;

      if (orderContainer && totalElement) {
        orderContainer.insertBefore(orderItemElement, totalElement);
      }
    }

    const clonedElement = clone.firstElementChild;
    const useElement = clone.querySelector("#scooter-icon");

    if (data.type === "delivery") {
      clonedElement.classList.add("delivery");
      const locationElement = clone.querySelector(".location");
      locationElement.textContent = data.location;

      useElement.setAttribute(
        "xlink:href",
        "./assets/icons/new-icon.svg#icon-new-view"
      );
    } else if (data.type === "dine-in") {
      clonedElement.classList.add("dine-in");
      const tableNoElement = clone.querySelector(".table-no");
      tableNoElement.textContent = data.tableNumber;
      useElement.setAttribute(
        "xlink:href",
        "./assets/icons/vivid-shop.svg#yellow-shop"
      );
    } else if (data.type === "takeaway") {
      clonedElement.classList.add("takeaway");
      useElement.setAttribute(
        "xlink:href",
        "./assets/icons/vivid-star.svg#moving-star"
      );
    }
    const dropdownButton = clone.querySelector(".order-dropdown-button");
    if (dropdownButton) {
      dropdownButton.addEventListener("click", () => {
        clonedElement.classList.toggle("open");
      });
    }

    idElement.addEventListener("click", () => {
      populateOrderModal(data.id);
    });

    customerNameElement.addEventListener("click", () => {
      populateCustomerProfileModal(data.customer.id);
    });

    container.appendChild(clone);
  }

  populateContainer();
};

const populateTransitOrder = (data) => {
  const container = document.querySelector(
    "#home .containers .transit-orders .orders"
  );
  const deliveryOrderTemplate = document.querySelector(
    "#transit-order-template"
  );

  function populateContainer(item) {
    container
      .querySelectorAll(":not(template)")
      .forEach((child) => child.remove());

    const clone = deliveryOrderTemplate.content.cloneNode(true);
    const idElement = clone.querySelector(".transit-order-id");
    const partnerAvatarElement = clone.querySelector(".partner-profile");
    const partnerNameElement = clone.querySelector(".order-name");
    const dateElement = clone.querySelector(".order-date");
    const timeCreatedElement = clone.querySelector(".order-time");

    idElement.textContent = item.id;
    partnerAvatarElement.src = item.partner.avatar;
    partnerNameElement.textContent = item.partner.name;
    dateElement.textContent = formatDate(item.dateCreated);
    timeCreatedElement.textContent = formatISODate(item.dateCreated);

    partnerNameElement.addEventListener("click", () => {
      populateDeliveryPartnerModal(item.id);
    });

    container.appendChild(clone);
  }

  data.forEach((item) => {
    populateContainer(item);
  });
};

const populateOrderContainers = async () => {
  const restaurantOrders = await getFiltered("orders", {
    businessID: restaurantID,
  });
  let deliveryOrders = restaurantOrders.filter((item) =>
    Object.entries({ type: "delivery" }).every(
      ([key, value]) => item[key] === value
    )
  );
  let ordersInTransit = deliveryOrders.filter((item) =>
    Object.entries({ orderStatus: "transit" }).every(
      ([key, value]) => item[key] === value
    )
  );

  let mostRecentOrder = restaurantOrders.reduce(
    (mostRecentItem, currentItem) => {
      const currentDate = new Date(currentItem.dateCreated);
      const mostRecentDate = new Date(mostRecentItem.dateCreated);

      if (isNaN(currentDate.getTime())) {
        throw new Error(
          `Invalid date in currentItem: ${JSON.stringify(currentItem)}`
        );
      }
      return currentDate > mostRecentDate ? currentItem : mostRecentItem;
    },
    restaurantOrders[0]
  );

  populateRecentOrder(mostRecentOrder);
  populateTransitOrder(ordersInTransit);
};

const populateHomePage = async () => {
  const topNavbar = document.querySelector(".top-nav");
  const response = await get("restaurants", restaurantID);
  restaurantData = response;
  // Set Notifications
  await populateNotifications();

  // Set Earnings
  populateEarnings();

  // Set Avatar
  const restaurantAvatars = topNavbar.querySelectorAll(
    ".profile .avatar, .user .avatar"
  );
  restaurantAvatars.forEach((item) => {
    item.src = restaurantData.businessAvatar;
  });

  // Set Name
  const ownersName = topNavbar.querySelector(
    ".notification-panel .profile .name"
  );
  ownersName.textContent = restaurantData.ownersName;

  // Set Restaurant Status
  const restaurantStatusBtns = topNavbar.querySelectorAll(".toggle-button");
  restaurantStatusBtns.forEach((button) => {
    const buttonText = button.querySelector(".status");
    const buttonSwitch = button.querySelector("img");
    try {
      if (restaurantData.businessStatus === "open") {
        button.classList.add("open");
        button.classList.remove("close");
        buttonText.textContent = "Open";
        buttonSwitch.src = "./assets/icons/toggle-on.svg";
      } else if (restaurantData.businessStatus === "close") {
        button.classList.replace("close", "open");
        buttonText.textContent = "Close";
        buttonSwitch.src = "./assets/icons/toggle-off.svg";
      }
    } catch (error) {
      createAlert(
        "couldn't update restaurant status",
        "./assets/icons/bin.svg",
        "red-5"
      );
    }
  });

  // Populate Information Cards
  populateInfoCards();

  // Populate Orders Containers
  populateOrderContainers();
};

const menuTypeImages = {
  "non-vegetarian": {
    icon: "./assets/icons/menu-type.svg#icon-non-vegetarian",
    name: "Non-vegetarian",
  },
  vegetarian: {
    icon: "./assets/icons/menu-type.svg#icon-vegetarian",
    name: "Vegetarian",
  },
};

const populateDishesInfoCard = () => {
  const dishesInfoCard = document.querySelector(
    "#dishes .categorized-container .info-card"
  );
  const dishesMenuInfo = dishesInfoCard.querySelector(
    "div > p:nth-child(1) > span:nth-child(2)"
  );
  const dishesDishInfo = dishesInfoCard.querySelector(
    "div > p:nth-child(2) > span:nth-child(2)"
  );

  // Count dishes
  dishesDishInfo.innerHTML = countAllDishes(menuData);
  dishesMenuInfo.innerHTML = menuData.categories.length;
};

const populateMenuCategoryTable = (data) => {
  const dishesTable = document.querySelector(
    "#dishes .categorized-container table tbody"
  );

  dishesTable.innerHTML = "";
  data.forEach((item, index) => {
    const row = document.createElement("tr");
    row.innerHTML = `
        <td>
          <label>
            <input type="checkbox" />
            <svg
              width="20"
              height="20"
              viewBox="0 0 20 20"
              fill="none"
              xmlns="http://www.w3.org/2000/svg"
            >
              <path
                d="M10 0.25C8.07164 0.25 6.18657 0.821828 4.58319 1.89317C2.97982 2.96451 1.73013 4.48726 0.992179 6.26884C0.254225 8.05042 0.061142 10.0108 0.437348 11.9021C0.813554 13.7934 1.74215 15.5307 3.10571 16.8943C4.46928 18.2579 6.20656 19.1865 8.09787 19.5627C9.98919 19.9389 11.9496 19.7458 13.7312 19.0078C15.5127 18.2699 17.0355 17.0202 18.1068 15.4168C19.1782 13.8134 19.75 11.9284 19.75 10C19.7473 7.41498 18.7192 4.93661 16.8913 3.10872C15.0634 1.28084 12.585 0.25273 10 0.25ZM14.2806 8.28063L9.03063 13.5306C8.96097 13.6004 8.87826 13.6557 8.78721 13.6934C8.69616 13.7312 8.59857 13.7506 8.5 13.7506C8.40144 13.7506 8.30385 13.7312 8.2128 13.6934C8.12175 13.6557 8.03903 13.6004 7.96938 13.5306L5.71938 11.2806C5.57865 11.1399 5.49959 10.949 5.49959 10.75C5.49959 10.551 5.57865 10.3601 5.71938 10.2194C5.86011 10.0786 6.05098 9.99958 6.25 9.99958C6.44903 9.99958 6.6399 10.0786 6.78063 10.2194L8.5 11.9397L13.2194 7.21937C13.2891 7.14969 13.3718 7.09442 13.4628 7.0567C13.5539 7.01899 13.6515 6.99958 13.75 6.99958C13.8486 6.99958 13.9461 7.01899 14.0372 7.0567C14.1282 7.09442 14.2109 7.14969 14.2806 7.21937C14.3503 7.28906 14.4056 7.37178 14.4433 7.46283C14.481 7.55387 14.5004 7.65145 14.5004 7.75C14.5004 7.84855 14.481 7.94613 14.4433 8.03717C14.4056 8.12822 14.3503 8.21094 14.2806 8.28063Z"
                fill="none"
                fill-opacity="0.6"
              />
            </svg>
          </label>
        </td>
        <td>${index + 1}</td>
        <td>${item.name}</td>
        <td>
          <svg width="18" height="19" fill="none">
            <use
              xlink:href="${menuTypeImages[item.type].icon}"
            ></use>
          </svg>
          <span>${menuTypeImages[item.type].name}</span>
        </td>
        <td>15 May, 2020 8:30am</td>
        <td>${item.availability ? "Available" : "Unavailable"}</td>
        <td class="enabled-cell">
          <button class="${item.status === "enabled" ? "enabled" : "disabled"}">
            <span class="button-text">${item.status}</span>
            <img
              src="${
                item.status === "enabled"
                  ? "./assets/icons/toggle-on.svg"
                  : "./assets/icons/toggle-off.svg"
              }"
              alt="toggle-button"
            />
          </button>
        </td>
        <td>
          <div class="static-dropdown">
            <button class="dropdown-button">
              <svg
                width="16"
                height="6"
                viewBox="0 0 22 6"
                fill="none"
                xmlns="http://www.w3.org/2000/svg"
              >
                <path
                  d="M13.625 3C13.625 3.51918 13.471 4.02669 13.1826 4.45837C12.8942 4.89005 12.4842 5.2265 12.0045 5.42518C11.5249 5.62386 10.9971 5.67585 10.4879 5.57456C9.97869 5.47328 9.51096 5.22327 9.14385 4.85616C8.77673 4.48904 8.52673 4.02131 8.42544 3.51211C8.32415 3.00291 8.37614 2.47511 8.57482 1.99546C8.7735 1.5158 9.10995 1.10583 9.54163 0.817392C9.97331 0.528953 10.4808 0.375 11 0.375C11.6962 0.375 12.3639 0.651562 12.8562 1.14384C13.3484 1.63613 13.625 2.30381 13.625 3ZM3.5 0.375C2.98083 0.375 2.47331 0.528953 2.04163 0.817392C1.60995 1.10583 1.2735 1.5158 1.07482 1.99546C0.876137 2.47511 0.824154 3.00291 0.92544 3.51211C1.02673 4.02131 1.27673 4.48904 1.64385 4.85616C2.01096 5.22327 2.47869 5.47328 2.98789 5.57456C3.49709 5.67585 4.02489 5.62386 4.50455 5.42518C4.9842 5.2265 5.39417 4.89005 5.68261 4.45837C5.97105 4.02669 6.125 3.51918 6.125 3C6.125 2.30381 5.84844 1.63613 5.35616 1.14384C4.86387 0.651562 4.19619 0.375 3.5 0.375ZM18.5 0.375C17.9808 0.375 17.4733 0.528953 17.0416 0.817392C16.61 1.10583 16.2735 1.5158 16.0748 1.99546C15.8761 2.47511 15.8242 3.00291 15.9254 3.51211C16.0267 4.02131 16.2767 4.48904 16.6438 4.85616C17.011 5.22327 17.4787 5.47328 17.9879 5.57456C18.4971 5.67585 19.0249 5.62386 19.5045 5.42518C19.9842 5.2265 20.3942 4.89005 20.6826 4.45837C20.971 4.02669 21.125 3.51918 21.125 3C21.125 2.65528 21.0571 2.31394 20.9252 1.99546C20.7933 1.67698 20.5999 1.3876 20.3562 1.14384C20.1124 0.900091 19.823 0.706735 19.5045 0.574816C19.1861 0.442897 18.8447 0.375 18.5 0.375Z"
                  fill="#FF0828"
                />
              </svg>
            </button>
            <div class="dropdown">
              <div class="dropdown-heading">
                <span>Action</span>
              </div>
              <ul>
                <li class="edit-button">
                  <span>Edit</span>
                </li>
                <li class="delete-button">
                  <span>Delete</span>
                  <svg
                    width="12"
                    height="13"
                    viewBox="0 0 12 13"
                    fill="none"
                    xmlns="http://www.w3.org/2000/svg"
                  >
                    <path
                      d="M11.5 2H9V1.5C9 1.10218 8.84196 0.720644 8.56066 0.43934C8.27936 0.158035 7.89782 0 7.5 0H4.5C4.10218 0 3.72064 0.158035 3.43934 0.43934C3.15804 0.720644 3 1.10218 3 1.5V2H0.5C0.367392 2 0.240215 2.05268 0.146447 2.14645C0.0526785 2.24021 0 2.36739 0 2.5C0 2.63261 0.0526785 2.75979 0.146447 2.85355C0.240215 2.94732 0.367392 3 0.5 3H1V12C1 12.2652 1.10536 12.5196 1.29289 12.7071C1.48043 12.8946 1.73478 13 2 13H10C10.2652 13 10.5196 12.8946 10.7071 12.7071C10.8946 12.5196 11 12.2652 11 12V3H11.5C11.6326 3 11.7598 2.94732 11.8536 2.85355C11.9473 2.75979 12 2.63261 12 2.5C12 2.36739 11.9473 2.24021 11.8536 2.14645C11.7598 2.05268 11.6326 2 11.5 2ZM5 9.5C5 9.63261 4.94732 9.75979 4.85355 9.85355C4.75979 9.94732 4.63261 10 4.5 10C4.36739 10 4.24021 9.94732 4.14645 9.85355C4.05268 9.75979 4 9.63261 4 9.5V5.5C4 5.36739 4.05268 5.24021 4.14645 5.14645C4.24021 5.05268 4.36739 5 4.5 5C4.63261 5 4.75979 5.05268 4.85355 5.14645C4.94732 5.24021 5 5.36739 5 5.5V9.5ZM8 9.5C8 9.63261 7.94732 9.75979 7.85355 9.85355C7.75979 9.94732 7.63261 10 7.5 10C7.36739 10 7.24021 9.94732 7.14645 9.85355C7.05268 9.75979 7 9.63261 7 9.5V5.5C7 5.36739 7.05268 5.24021 7.14645 5.14645C7.24021 5.05268 7.36739 5 7.5 5C7.63261 5 7.75979 5.05268 7.85355 5.14645C7.94732 5.24021 8 5.36739 8 5.5V9.5ZM8 2H4V1.5C4 1.36739 4.05268 1.24021 4.14645 1.14645C4.24021 1.05268 4.36739 1 4.5 1H7.5C7.63261 1 7.75979 1.05268 7.85355 1.14645C7.94732 1.24021 8 1.36739 8 1.5V2Z"
                      fill="#FF0828"
                    />
                  </svg>
                </li>
              </ul>
            </div>
          </div>
        </td>
      `;
    dishesTable.appendChild(row);

    const toggleSwitch = row.querySelector(".enabled-cell button");
    toggleSwitch.addEventListener("click", async () => {
      if (item.status == "enabled") {
        data[index].status = "disabled";
      } else if (item.status == "disabled") {
        data[index].status = "enabled";
      }
      try {
        let updatedData = {
          categories: menuData,
        };
        await updateIfHasValue(
          "menus",
          menuData.id,
          "businessID",
          restaurantID,
          updatedData
        );
        populateMenuCategoryTable(menuData.categories);
        populateCardsByCategory(menuData.categories);
        // const menuData = await getFiltered("menus", {
        //   businessID: restaurantID,
        // });
        // console.log("Try block", menuData);
      } catch (error) {
        createAlert(
          "couldn't edit category",
          "./assets/icons/bin.svg",
          "red-5"
        );
      }
    });

    const editButtons = row.querySelectorAll(
      "td:nth-child(3), td:nth-child(8) .static-dropdown li.edit-button"
    );
    editButtons.forEach((button) => {
      button.addEventListener("click", () => {
        editCategoryModal(item).then(async (newData) => {
          data = data.map((obj) => (obj.id === item.id ? newData : obj));
          const updatedDishItems = data[index].dishes.map((dishItem) =>
            dishItem.category === item.name.toLowerCase()
              ? { ...dishItem, category: newData.name, type: newData.type }
              : dishItem
          );
          try {
            data[index].dishes = updatedDishItems;
            menuData.categories = data;
            let updatedData = {
              categories: menuData,
            };
            await updateIfHasValue(
              "menus",
              menuData.id,
              "businessID",
              restaurantID,
              updatedData
            );
            populateMenuCategoryTable(menuData.categories);
            populateCardsByCategory(menuData.categories);

            // const menuData = await getFiltered("menus", {
            //   businessID: restaurantID,
            // });
            // console.log("Try block", menuData);
          } catch (error) {
            createAlert(
              "couldn't edit category",
              "./assets/icons/bin.svg",
              "red-5"
            );
          }
        });
      });
    });

    const deleteButton = row.querySelector(
      "td:nth-child(8) .static-dropdown li.delete-button"
    );
    deleteButton.addEventListener("click", () => {
      const existingModal = document.querySelector(".modal-wrapper");
      const modal = existingModal.cloneNode(true);

      const modalAlreadyExists = document.querySelector(
        "#delete-category-modal"
      );
      if (modalAlreadyExists) return;

      modal.id = "delete-category-modal";
      modal.classList.add("confirm-modal", "lined-modal");

      modal.innerHTML = `
          <div class="modal-heading">
            <p>
              <svg width="20" height="22" viewBox="0 0 20 22" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M18.25 4.5H14.875V3.375C14.875 2.67881 14.5984 2.01113 14.1062 1.51884C13.6139 1.02656 12.9462 0.75 12.25 0.75H7.75C7.05381 0.75 6.38613 1.02656 5.89384 1.51884C5.40156 2.01113 5.125 2.67881 5.125 3.375V4.5H1.75C1.45163 4.5 1.16548 4.61853 0.954505 4.8295C0.743526 5.04048 0.625 5.32663 0.625 5.625C0.625 5.92337 0.743526 6.20952 0.954505 6.4205C1.16548 6.63147 1.45163 6.75 1.75 6.75H2.125V19.5C2.125 19.9973 2.32254 20.4742 2.67417 20.8258C3.02581 21.1775 3.50272 21.375 4 21.375H16C16.4973 21.375 16.9742 21.1775 17.3258 20.8258C17.6775 20.4742 17.875 19.9973 17.875 19.5V6.75H18.25C18.5484 6.75 18.8345 6.63147 19.0455 6.4205C19.2565 6.20952 19.375 5.92337 19.375 5.625C19.375 5.32663 19.2565 5.04048 19.0455 4.8295C18.8345 4.61853 18.5484 4.5 18.25 4.5ZM7.375 3.375C7.375 3.27554 7.41451 3.18016 7.48483 3.10984C7.55516 3.03951 7.65054 3 7.75 3H12.25C12.3495 3 12.4448 3.03951 12.5152 3.10984C12.5855 3.18016 12.625 3.27554 12.625 3.375V4.5H7.375V3.375ZM15.625 19.125H4.375V6.75H15.625V19.125ZM8.875 9.75V15.75C8.875 16.0484 8.75647 16.3345 8.5455 16.5455C8.33452 16.7565 8.04837 16.875 7.75 16.875C7.45163 16.875 7.16548 16.7565 6.9545 16.5455C6.74353 16.3345 6.625 16.0484 6.625 15.75V9.75C6.625 9.45163 6.74353 9.16548 6.9545 8.9545C7.16548 8.74353 7.45163 8.625 7.75 8.625C8.04837 8.625 8.33452 8.74353 8.5455 8.9545C8.75647 9.16548 8.875 9.45163 8.875 9.75ZM13.375 9.75V15.75C13.375 16.0484 13.2565 16.3345 13.0455 16.5455C12.8345 16.7565 12.5484 16.875 12.25 16.875C11.9516 16.875 11.6655 16.7565 11.4545 16.5455C11.2435 16.3345 11.125 16.0484 11.125 15.75V9.75C11.125 9.45163 11.2435 9.16548 11.4545 8.9545C11.6655 8.74353 11.9516 8.625 12.25 8.625C12.5484 8.625 12.8345 8.74353 13.0455 8.9545C13.2565 9.16548 13.375 9.45163 13.375 9.75Z" fill="#2D2C2C"/>
              </svg>
              <span>Delete?</span>
            </p>
          </div>
          <div class="modal-body">
            <span>Oh No!👊🏾</span>
            <h2>Are you sure you want this item deleted? </h2>
            <label class="checkbox">
              <input
                type="checkbox"
                id="delete-category-dishes"
                value="delete-category-dishes"
              />
              <svg
                width="20"
                height="20"
                viewBox="0 0 20 20"
                fill="none"
                xmlns="http://www.w3.org/2000/svg"
              >
                <path
                  d="M10 0.25C8.07164 0.25 6.18657 0.821828 4.58319 1.89317C2.97982 2.96451 1.73013 4.48726 0.992179 6.26884C0.254225 8.05042 0.061142 10.0108 0.437348 11.9021C0.813554 13.7934 1.74215 15.5307 3.10571 16.8943C4.46928 18.2579 6.20656 19.1865 8.09787 19.5627C9.98919 19.9389 11.9496 19.7458 13.7312 19.0078C15.5127 18.2699 17.0355 17.0202 18.1068 15.4168C19.1782 13.8134 19.75 11.9284 19.75 10C19.7473 7.41498 18.7192 4.93661 16.8913 3.10872C15.0634 1.28084 12.585 0.25273 10 0.25ZM14.2806 8.28063L9.03063 13.5306C8.96097 13.6004 8.87826 13.6557 8.78721 13.6934C8.69616 13.7312 8.59857 13.7506 8.5 13.7506C8.40144 13.7506 8.30385 13.7312 8.2128 13.6934C8.12175 13.6557 8.03903 13.6004 7.96938 13.5306L5.71938 11.2806C5.57865 11.1399 5.49959 10.949 5.49959 10.75C5.49959 10.551 5.57865 10.3601 5.71938 10.2194C5.86011 10.0786 6.05098 9.99958 6.25 9.99958C6.44903 9.99958 6.6399 10.0786 6.78063 10.2194L8.5 11.9397L13.2194 7.21937C13.2891 7.14969 13.3718 7.09442 13.4628 7.0567C13.5539 7.01899 13.6515 6.99958 13.75 6.99958C13.8486 6.99958 13.9461 7.01899 14.0372 7.0567C14.1282 7.09442 14.2109 7.14969 14.2806 7.21937C14.3503 7.28906 14.4056 7.37178 14.4433 7.46283C14.481 7.55387 14.5004 7.65145 14.5004 7.75C14.5004 7.84855 14.481 7.94613 14.4433 8.03717C14.4056 8.12822 14.3503 8.21094 14.2806 8.28063Z"
                  fill="none"
                  fill-opacity="0.6"
                />
              </svg>
              <span>Delete the dishes in this category</span>
            </label>
            <p class="button-section">
            <button id="confirm-delete">
              Delete
            </button>
            <button id="cancel-button">Cancel</button>
            </p>
          </div>
        `;
      main.appendChild(modal);
      modal.showModal();

      const cancelButton = modal.querySelector("#cancel-button");
      cancelButton.addEventListener("click", () => {
        modal.close();
        modal.remove();
      });

      const confirmDeleteButton = modal.querySelector("#confirm-delete");
      confirmDeleteButton.addEventListener("click", async () => {
        let uncategorizedDishes = item.dishes;
        for (let i = data.length - 1; i >= 0; i--) {
          if (data[i].id === item.id) {
            data.splice(i, 1);
          }
        }
        try {
          let updatedData;
          let isChecked = modal.querySelector(
            "#delete-category-dishes"
          ).checked;
          if (isChecked) {
            updatedData = {
              categories: menuData,
            };
          } else {
            menuData.uncategorizedItems.push(...uncategorizedDishes);
            updatedData = {
              categories: menuData,
              uncategorizedItems: menuData.uncategorizedItems,
            };
          }
          await updateIfHasValue(
            "menus",
            menuData.id,
            "businessID",
            restaurantID,
            updatedData
          );
          createAlert("deleted", "./assets/icons/bin.svg", "red-5");
          setTimeout(() => {
            modal.close();
            modal.remove();
            setTimeout(() => {
              populateMenuCategoryTable(menuData.categories);
              populateCardsByCategory(menuData.categories);
              populateUncategorizedContainer(menuData.uncategorizedItems);
              populateInfoCards();
              populateDishesInfoCard();
            }, 500);
          }, 800);
        } catch (error) {
          createAlert(
            "couldn't delete category",
            "./assets/icons/bin.svg",
            "red-5"
          );
        }
      });
    });
  });
};

function populateCardsByCategory(data) {
  let container = dishesPage.querySelector(".categorized-container .dishes");
  container.textContent = "";
  data.forEach((category, index) => {
    const categoryContainerId = `category-${category.name}`;
    let categoryContainer = document.getElementById(categoryContainerId);

    if (!categoryContainer) {
      categoryContainer = document.createElement("div");
      categoryContainer.id = categoryContainerId;
      categoryContainer.classList.add("dishes-section");

      const header = document.createElement("div");
      header.classList.add("dishes-header");
      header.innerHTML = `
        <div>
          <svg
            width="16"
            height="16"
            viewBox="0 0 18 18"
            fill="none"
            xmlns="http://www.w3.org/2000/svg"
          >
            <path
              d="M8.25 4.5C8.25 5.24168 8.03007 5.96671 7.61801 6.58339C7.20596 7.20007 6.62029 7.68072 5.93506 7.96455C5.24984 8.24838 4.49584 8.32264 3.76841 8.17795C3.04098 8.03325 2.3728 7.6761 1.84835 7.15165C1.3239 6.6272 0.966751 5.95902 0.822057 5.23159C0.677362 4.50416 0.751625 3.75016 1.03545 3.06494C1.31928 2.37972 1.79993 1.79405 2.41661 1.38199C3.0333 0.969935 3.75832 0.750002 4.5 0.750002C5.49456 0.750002 6.44839 1.14509 7.15165 1.84835C7.85491 2.55161 8.25 3.50544 8.25 4.5ZM13.5 8.25C14.2417 8.25 14.9667 8.03007 15.5834 7.61801C16.2001 7.20596 16.6807 6.62029 16.9646 5.93506C17.2484 5.24984 17.3226 4.49584 17.1779 3.76841C17.0333 3.04098 16.6761 2.3728 16.1517 1.84835C15.6272 1.3239 14.959 0.966751 14.2316 0.822057C13.5042 0.677362 12.7502 0.751625 12.0649 1.03545C11.3797 1.31928 10.794 1.79993 10.382 2.41661C9.96993 3.0333 9.75 3.75832 9.75 4.5C9.75 5.49456 10.1451 6.44839 10.8484 7.15165C11.5516 7.85491 12.5054 8.25 13.5 8.25ZM4.5 9.75C3.75832 9.75 3.0333 9.96993 2.41661 10.382C1.79993 10.794 1.31928 11.3797 1.03545 12.0649C0.751625 12.7502 0.677362 13.5042 0.822057 14.2316C0.966751 14.959 1.3239 15.6272 1.84835 16.1517C2.3728 16.6761 3.04098 17.0333 3.76841 17.1779C4.49584 17.3226 5.24984 17.2484 5.93506 16.9646C6.62029 16.6807 7.20596 16.2001 7.61801 15.5834C8.03007 14.9667 8.25 14.2417 8.25 13.5C8.25 12.5054 7.85491 11.5516 7.15165 10.8484C6.44839 10.1451 5.49456 9.75 4.5 9.75ZM13.5 9.75C12.7583 9.75 12.0333 9.96993 11.4166 10.382C10.7999 10.794 10.3193 11.3797 10.0355 12.0649C9.75162 12.7502 9.67736 13.5042 9.82206 14.2316C9.96675 14.959 10.3239 15.6272 10.8484 16.1517C11.3728 16.6761 12.041 17.0333 12.7684 17.1779C13.4958 17.3226 14.2498 17.2484 14.9351 16.9646C15.6203 16.6807 16.206 16.2001 16.618 15.5834C17.0301 14.9667 17.25 14.2417 17.25 13.5C17.25 12.5054 16.8549 11.5516 16.1517 10.8484C15.4484 10.1451 14.4946 9.75 13.5 9.75Z"
              fill="#2D2C2C"
              fill-opacity="0.4"
            />
          </svg>
          <span>${category.name}</span>
          <button class="toggle-button ${
            category.status === "enabled" ? "enabled" : "disabled"
          }">
              <span class="button-text">${
                category.status === "enabled" ? "Enabled" : "Disabled"
              }</span>
              <img
                src="${
                  category.status === "enabled"
                    ? "./assets/icons/toggle-on.svg"
                    : "./assets/icons/toggle-off.svg"
                }"
                alt="toggle-button"
              />
            </button>
        </div>
        <div class="static-dropdown today">
          <button class="dropdown-button">
            <svg
              width="18"
              height="8"
              viewBox="0 0 28 8"
              fill="none"
              xmlns="http://www.w3.org/2000/svg"
            >
              <path
                d="M17.5 4C17.5 4.69223 17.2947 5.36892 16.9101 5.9445C16.5256 6.52007 15.9789 6.96867 15.3394 7.23358C14.6999 7.49848 13.9961 7.5678 13.3172 7.43275C12.6383 7.2977 12.0146 6.96436 11.5251 6.47487C11.0356 5.98539 10.7023 5.36175 10.5673 4.68282C10.4322 4.00388 10.5015 3.30015 10.7664 2.66061C11.0313 2.02107 11.4799 1.47444 12.0555 1.08986C12.6311 0.705271 13.3078 0.5 14 0.5C14.9283 0.5 15.8185 0.868749 16.4749 1.52513C17.1313 2.1815 17.5 3.07174 17.5 4ZM4 0.5C3.30777 0.5 2.63108 0.705271 2.05551 1.08986C1.47993 1.47444 1.03133 2.02107 0.766423 2.66061C0.501516 3.30015 0.432205 4.00388 0.567253 4.68282C0.702301 5.36175 1.03564 5.98539 1.52513 6.47487C2.01461 6.96436 2.63825 7.2977 3.31719 7.43275C3.99612 7.5678 4.69985 7.49848 5.33939 7.23358C5.97894 6.96867 6.52556 6.52007 6.91015 5.9445C7.29473 5.36892 7.5 4.69223 7.5 4C7.5 3.07174 7.13125 2.1815 6.47487 1.52513C5.8185 0.868749 4.92826 0.5 4 0.5ZM24 0.5C23.3078 0.5 22.6311 0.705271 22.0555 1.08986C21.4799 1.47444 21.0313 2.02107 20.7664 2.66061C20.5015 3.30015 20.4322 4.00388 20.5673 4.68282C20.7023 5.36175 21.0356 5.98539 21.5251 6.47487C22.0146 6.96436 22.6383 7.2977 23.3172 7.43275C23.9961 7.5678 24.6999 7.49848 25.3394 7.23358C25.9789 6.96867 26.5256 6.52007 26.9101 5.9445C27.2947 5.36892 27.5 4.69223 27.5 4C27.5 3.54037 27.4095 3.08525 27.2336 2.66061C27.0577 2.23597 26.7999 1.85013 26.4749 1.52513C26.1499 1.20012 25.764 0.942313 25.3394 0.766421C24.9148 0.59053 24.4596 0.5 24 0.5Z"
                fill="#2D2C2C"
              />
            </svg>
          </button>
          <div class="dropdown">
            <div class="dropdown-heading">
              <span>Action</span>
            </div>
            <ul>
              <li class="edit-category-button">
                <span>Edit</span>
              </li>
              <li class="make-available-button">
                <span>Make all Available</span>
              </li>
              <li class="make-unavailable-button">
                <span>Make all Unavailable</span>
              </li>
              <li class="increase-price">
                <span>Increase Price</span>
              </li>
              <li class="decrease-price">
                <span>Decrease Price</span>
              </li>
              <li class="rearrange-category-button">
                <span>Rearrange Menu Categories</span>
              </li>
              <li class="rearrange-dishes-button">
                <span>Rearrange Dish</span>
              </li>
              <li class="delete-button">
                <span>Delete</span>
                <svg
                  width="12"
                  height="13"
                  viewBox="0 0 12 13"
                  fill="none"
                  xmlns="http://www.w3.org/2000/svg"
                >
                  <path
                    d="M11.5 2H9V1.5C9 1.10218 8.84196 0.720644 8.56066 0.43934C8.27936 0.158035 7.89782 0 7.5 0H4.5C4.10218 0 3.72064 0.158035 3.43934 0.43934C3.15804 0.720644 3 1.10218 3 1.5V2H0.5C0.367392 2 0.240215 2.05268 0.146447 2.14645C0.0526785 2.24021 0 2.36739 0 2.5C0 2.63261 0.0526785 2.75979 0.146447 2.85355C0.240215 2.94732 0.367392 3 0.5 3H1V12C1 12.2652 1.10536 12.5196 1.29289 12.7071C1.48043 12.8946 1.73478 13 2 13H10C10.2652 13 10.5196 12.8946 10.7071 12.7071C10.8946 12.5196 11 12.2652 11 12V3H11.5C11.6326 3 11.7598 2.94732 11.8536 2.85355C11.9473 2.75979 12 2.63261 12 2.5C12 2.36739 11.9473 2.24021 11.8536 2.14645C11.7598 2.05268 11.6326 2 11.5 2ZM5 9.5C5 9.63261 4.94732 9.75979 4.85355 9.85355C4.75979 9.94732 4.63261 10 4.5 10C4.36739 10 4.24021 9.94732 4.14645 9.85355C4.05268 9.75979 4 9.63261 4 9.5V5.5C4 5.36739 4.05268 5.24021 4.14645 5.14645C4.24021 5.05268 4.36739 5 4.5 5C4.63261 5 4.75979 5.05268 4.85355 5.14645C4.94732 5.24021 5 5.36739 5 5.5V9.5ZM8 9.5C8 9.63261 7.94732 9.75979 7.85355 9.85355C7.75979 9.94732 7.63261 10 7.5 10C7.36739 10 7.24021 9.94732 7.14645 9.85355C7.05268 9.75979 7 9.63261 7 9.5V5.5C7 5.36739 7.05268 5.24021 7.14645 5.14645C7.24021 5.05268 7.36739 5 7.5 5C7.63261 5 7.75979 5.05268 7.85355 5.14645C7.94732 5.24021 8 5.36739 8 5.5V9.5ZM8 2H4V1.5C4 1.36739 4.05268 1.24021 4.14645 1.14645C4.24021 1.05268 4.36739 1 4.5 1H7.5C7.63261 1 7.75979 1.05268 7.85355 1.14645C7.94732 1.24021 8 1.36739 8 1.5V2Z"
                    fill="#FF0828"
                  />
                </svg>
              </li>
            </ul>
          </div>
        </div>
      `;
      categoryContainer.appendChild(header);

      const dishesBody = document.createElement("div");
      dishesBody.classList.add("dishes-body");
      categoryContainer.appendChild(dishesBody);

      container.appendChild(categoryContainer);

      const editCategoryBtn = header.querySelector(
        ".static-dropdown .edit-category-button"
      );
      editCategoryBtn.addEventListener("click", () => {
        editCategoryModal(category).then(async (data) => {
          if (data) {
            let editedData = menuData.categories.map((obj) =>
              obj.id === data.id ? data : obj
            );
            try {
              menuData.categories = editedData;
              let updatedData = {
                categories: menuData,
              };
              await updateIfHasValue(
                "menus",
                menuData.id,
                "businessID",
                restaurantID,
                updatedData
              );
              populateMenuCategoryTable(menuData.categories);
              populateCardsByCategory(menuData.categories);

              // const menuData = await getFiltered("menus", {
              //   businessID: restaurantID,
              // });
              // console.log("Try block", menuData);
            } catch (error) {
              createAlert(
                "couldn't edit category",
                "./assets/icons/bin.svg",
                "red-5"
              );
            }
          }
        });
      });

      const increasePriceBtn = header.querySelector(
        ".static-dropdown .increase-price"
      );
      increasePriceBtn.addEventListener("click", async () => {
        category.dishes = category.dishes.map((item) => ({
          ...item,
          price: {
            ...item.price,
            regularPrice: +(item.price.regularPrice * 1.05).toFixed(2), // Increases price by 5%
          },
        }));
        try {
          menuData.categories[index] = category;
          let updatedData = {
            categories: menuData,
          };
          await updateIfHasValue(
            "menus",
            menuData.id,
            "businessID",
            restaurantID,
            updatedData
          );
          populateMenuCategoryTable(menuData.categories);
          populateCardsByCategory(menuData.categories);
        } catch (error) {
          createAlert(
            "couldn't edit category",
            "./assets/icons/bin.svg",
            "red-5"
          );
        }
      });

      const decreasePriceBtn = header.querySelector(
        ".static-dropdown .decrease-price"
      );
      decreasePriceBtn.addEventListener("click", async () => {
        category.dishes = category.dishes.map((item) => ({
          ...item,
          price: {
            ...item.price,
            regularPrice: +(item.price.regularPrice * 0.95).toFixed(2), // Increases price by 5%
          },
        }));
        try {
          menuData.categories[index] = category;
          let updatedData = {
            categories: menuData,
          };
          await updateIfHasValue(
            "menus",
            menuData.id,
            "businessID",
            restaurantID,
            updatedData
          );
          populateMenuCategoryTable(menuData.categories);
          populateCardsByCategory(menuData.categories);
        } catch (error) {
          createAlert(
            "couldn't edit category",
            "./assets/icons/bin.svg",
            "red-5"
          );
        }
      });

      const makeAvailableBtn = header.querySelector(
        ".static-dropdown .make-available-button"
      );
      makeAvailableBtn.addEventListener("click", () => {
        menuData.categories[index].dishes.forEach(async (dish) => {
          const updatedCategories = menuData.categories.map((obj) =>
            obj.id === dish.id ? { ...obj, availability: true } : obj
          );
          menuData.categories = updatedCategories;
          try {
            let updatedData = {
              categories: menuData.categories,
            };
            await updateIfHasValue(
              "menus",
              menuData.id,
              "businessID",
              restaurantID,
              updatedData
            );
            populateMenuCategoryTable(menuData.categories);
            populateCardsByCategory(menuData.categories);
          } catch (error) {
            createAlert(
              "couldn't edit category",
              "./assets/icons/bin.svg",
              "red-5"
            );
          }
        });
      });

      const makeUnavailableBtn = header.querySelector(
        ".static-dropdown .make-unavailable-button"
      );
      makeUnavailableBtn.addEventListener("click", () => {
        menuData.categories[index].dishes.forEach(async (dish) => {
          const updatedCategories = menuData.categories.map((obj) =>
            obj.id === dish.id ? { ...obj, availability: false } : obj
          );
          menuData.categories = updatedCategories;
          try {
            let updatedData = {
              categories: menuData.categories,
            };
            await updateIfHasValue(
              "menus",
              menuData.id,
              "businessID",
              restaurantID,
              updatedData
            );
            populateMenuCategoryTable(menuData.categories);
            populateCardsByCategory(menuData.categories);
          } catch (error) {
            createAlert(
              "couldn't edit category",
              "./assets/icons/bin.svg",
              "red-5"
            );
          }
        });
      });

      const rearrangeCategoryBtn = header.querySelector(
        ".static-dropdown .rearrange-category-button"
      );
      rearrangeCategoryBtn.addEventListener("click", () => {
        rearrangeCategoryModal(menuData.categories).then(
          async (rearrangedData) => {
            if (rearrangedData) {
              try {
                menuData.categories = rearrangedData;
                let updatedData = {
                  categories: menuData.categories,
                };
                await updateIfHasValue(
                  "menus",
                  menuData.id,
                  "businessID",
                  restaurantID,
                  updatedData
                );
                populateMenuCategoryTable(menuData.categories);
                populateCardsByCategory(menuData.categories);
              } catch (error) {
                createAlert(
                  "couldn't edit category",
                  "./assets/icons/bin.svg",
                  "red-5"
                );
              }
            } else {
              console.log("Modal closed without saving.");
            }
          }
        );
      });

      const rearrangeDishesBtn = header.querySelector(
        ".static-dropdown .rearrange-dishes-button"
      );
      rearrangeDishesBtn.addEventListener("click", () => {
        console.log();
        rearrangeDishesModal(
          category.name,
          menuData.categories[index].dishes
        ).then(async (rearrangedData) => {
          if (rearrangedData) {
            try {
              menuData.categories[index].dishes = rearrangedData;
              let updatedData = {
                categories: menuData.categories,
              };
              await updateIfHasValue(
                "menus",
                menuData.id,
                "businessID",
                restaurantID,
                updatedData
              );
              populateCardsByCategory(menuData.categories);
            } catch (error) {
              createAlert(
                "couldn't edit category",
                "./assets/icons/bin.svg",
                "red-5"
              );
            }
          } else {
            console.log("Modal closed without saving.");
          }
        });
      });

      const toggleSwitch = header.querySelector(".toggle-button");
      toggleSwitch.addEventListener("click", async () => {
        if (category.status == "enabled") {
          data[index].status = "disabled";
        } else if (category.status == "disabled") {
          data[index].status = "enabled";
        }
        try {
          let updatedData = {
            categories: menuData,
          };
          await updateIfHasValue(
            "menus",
            menuData.id,
            "businessID",
            restaurantID,
            updatedData
          );
          // menuData = response;
          populateMenuCategoryTable(menuData.categories);
          populateCardsByCategory(menuData.categories);
          // const menuData = await getFiltered("menus", {
          //   businessID: restaurantID,
          // });
          // console.log("Try block", menuData);
        } catch (error) {
          createAlert(
            "couldn't edit category",
            "./assets/icons/bin.svg",
            "red-5"
          );
        }
      });

      const categoryDeleteButton = header.querySelector(
        ".static-dropdown .delete-button"
      );
      categoryDeleteButton.addEventListener("click", () => {
        const existingModal = document.querySelector(".modal-wrapper");
        const modal = existingModal.cloneNode(true);

        const modalAlreadyExists = document.querySelector(
          "#delete-category-modal"
        );
        if (modalAlreadyExists) return;

        modal.id = "delete-category-modal";
        modal.classList.add("confirm-modal", "lined-modal");

        modal.innerHTML = `
            <div class="modal-heading">
              <p>
                <svg width="20" height="22" viewBox="0 0 20 22" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M18.25 4.5H14.875V3.375C14.875 2.67881 14.5984 2.01113 14.1062 1.51884C13.6139 1.02656 12.9462 0.75 12.25 0.75H7.75C7.05381 0.75 6.38613 1.02656 5.89384 1.51884C5.40156 2.01113 5.125 2.67881 5.125 3.375V4.5H1.75C1.45163 4.5 1.16548 4.61853 0.954505 4.8295C0.743526 5.04048 0.625 5.32663 0.625 5.625C0.625 5.92337 0.743526 6.20952 0.954505 6.4205C1.16548 6.63147 1.45163 6.75 1.75 6.75H2.125V19.5C2.125 19.9973 2.32254 20.4742 2.67417 20.8258C3.02581 21.1775 3.50272 21.375 4 21.375H16C16.4973 21.375 16.9742 21.1775 17.3258 20.8258C17.6775 20.4742 17.875 19.9973 17.875 19.5V6.75H18.25C18.5484 6.75 18.8345 6.63147 19.0455 6.4205C19.2565 6.20952 19.375 5.92337 19.375 5.625C19.375 5.32663 19.2565 5.04048 19.0455 4.8295C18.8345 4.61853 18.5484 4.5 18.25 4.5ZM7.375 3.375C7.375 3.27554 7.41451 3.18016 7.48483 3.10984C7.55516 3.03951 7.65054 3 7.75 3H12.25C12.3495 3 12.4448 3.03951 12.5152 3.10984C12.5855 3.18016 12.625 3.27554 12.625 3.375V4.5H7.375V3.375ZM15.625 19.125H4.375V6.75H15.625V19.125ZM8.875 9.75V15.75C8.875 16.0484 8.75647 16.3345 8.5455 16.5455C8.33452 16.7565 8.04837 16.875 7.75 16.875C7.45163 16.875 7.16548 16.7565 6.9545 16.5455C6.74353 16.3345 6.625 16.0484 6.625 15.75V9.75C6.625 9.45163 6.74353 9.16548 6.9545 8.9545C7.16548 8.74353 7.45163 8.625 7.75 8.625C8.04837 8.625 8.33452 8.74353 8.5455 8.9545C8.75647 9.16548 8.875 9.45163 8.875 9.75ZM13.375 9.75V15.75C13.375 16.0484 13.2565 16.3345 13.0455 16.5455C12.8345 16.7565 12.5484 16.875 12.25 16.875C11.9516 16.875 11.6655 16.7565 11.4545 16.5455C11.2435 16.3345 11.125 16.0484 11.125 15.75V9.75C11.125 9.45163 11.2435 9.16548 11.4545 8.9545C11.6655 8.74353 11.9516 8.625 12.25 8.625C12.5484 8.625 12.8345 8.74353 13.0455 8.9545C13.2565 9.16548 13.375 9.45163 13.375 9.75Z" fill="#2D2C2C"/>
                </svg>
                <span>Delete?</span>
              </p>
            </div>
            <div class="modal-body">
              <span>Oh No!👊🏾</span>
              <h2>Are you sure you want this item deleted? </h2>
              <label class="checkbox">
                <input
                  type="checkbox"
                  id="delete-category-dishes"
                  value="delete-category-dishes"
                />
                <svg
                  width="20"
                  height="20"
                  viewBox="0 0 20 20"
                  fill="none"
                  xmlns="http://www.w3.org/2000/svg"
                >
                  <path
                    d="M10 0.25C8.07164 0.25 6.18657 0.821828 4.58319 1.89317C2.97982 2.96451 1.73013 4.48726 0.992179 6.26884C0.254225 8.05042 0.061142 10.0108 0.437348 11.9021C0.813554 13.7934 1.74215 15.5307 3.10571 16.8943C4.46928 18.2579 6.20656 19.1865 8.09787 19.5627C9.98919 19.9389 11.9496 19.7458 13.7312 19.0078C15.5127 18.2699 17.0355 17.0202 18.1068 15.4168C19.1782 13.8134 19.75 11.9284 19.75 10C19.7473 7.41498 18.7192 4.93661 16.8913 3.10872C15.0634 1.28084 12.585 0.25273 10 0.25ZM14.2806 8.28063L9.03063 13.5306C8.96097 13.6004 8.87826 13.6557 8.78721 13.6934C8.69616 13.7312 8.59857 13.7506 8.5 13.7506C8.40144 13.7506 8.30385 13.7312 8.2128 13.6934C8.12175 13.6557 8.03903 13.6004 7.96938 13.5306L5.71938 11.2806C5.57865 11.1399 5.49959 10.949 5.49959 10.75C5.49959 10.551 5.57865 10.3601 5.71938 10.2194C5.86011 10.0786 6.05098 9.99958 6.25 9.99958C6.44903 9.99958 6.6399 10.0786 6.78063 10.2194L8.5 11.9397L13.2194 7.21937C13.2891 7.14969 13.3718 7.09442 13.4628 7.0567C13.5539 7.01899 13.6515 6.99958 13.75 6.99958C13.8486 6.99958 13.9461 7.01899 14.0372 7.0567C14.1282 7.09442 14.2109 7.14969 14.2806 7.21937C14.3503 7.28906 14.4056 7.37178 14.4433 7.46283C14.481 7.55387 14.5004 7.65145 14.5004 7.75C14.5004 7.84855 14.481 7.94613 14.4433 8.03717C14.4056 8.12822 14.3503 8.21094 14.2806 8.28063Z"
                    fill="none"
                    fill-opacity="0.6"
                  />
                </svg>
                <span>Delete the dishes in this category</span>
              </label>
              <p class="button-section">
              <button id="confirm-delete">
                Delete
              </button>
              <button id="cancel-button">Cancel</button>
              </p>
            </div>
          `;
        main.appendChild(modal);
        modal.showModal();

        const cancelButton = modal.querySelector("#cancel-button");
        cancelButton.addEventListener("click", () => {
          modal.close();
          modal.remove();
        });

        const deleteButton = modal.querySelector("#confirm-delete");
        deleteButton.addEventListener("click", async () => {
          let uncategorizedDishes = category.dishes;
          for (let i = data.length - 1; i >= 0; i--) {
            if (data[i].id === category.id) {
              data.splice(i, 1);
            }
          }
          try {
            let updatedData;
            let isChecked = modal.querySelector(
              "#delete-category-dishes"
            ).checked;
            if (isChecked) {
              updatedData = {
                categories: menuData,
              };
            } else {
              menuData.uncategorizedItems.push(...uncategorizedDishes);
              updatedData = {
                categories: menuData,
                uncategorizedItems: menuData.uncategorizedItems,
              };
            }
            await updateIfHasValue(
              "menus",
              menuData.id,
              "businessID",
              restaurantID,
              updatedData
            );
            createAlert("deleted", "./assets/icons/bin.svg", "red-5");
            setTimeout(() => {
              modal.close();
              modal.remove();
              setTimeout(() => {
                populateMenuCategoryTable(menuData.categories);
                populateCardsByCategory(menuData.categories);
                populateUncategorizedContainer(menuData.uncategorizedItems);
              }, 500);
            }, 800);
          } catch (error) {
            createAlert(
              "couldn't delete category",
              "./assets/icons/bin.svg",
              "red-5"
            );
          }
        });
      });
    }

    const dishesBody = categoryContainer.querySelector(".dishes-body");

    // Populate items in the category
    category.dishes.forEach((item, index) => {
      const card = document.createElement("div");
      card.classList.add("dish-card");
      if (item.status === "disabled") {
        card.classList.add("disabled");
      }
      card.innerHTML = `
          <div class="content">
            <p>
              <span class="title">
                ${item.name}
              </span>
              <span class="price"> $${item.price.regularPrice}</span>
            </p>
            <p class="button-section">
              <button class="edit-button">
                <svg
                  width="13"
                  height="13"
                  viewBox="0 0 16 16"
                  fill="none"
                  xmlns="http://www.w3.org/2000/svg"
                >
                  <path
                    d="M15.1824 4.45954L12.0402 1.31798C11.7765 1.05435 11.4188 0.90625 11.0459 0.90625C10.6731 0.90625 10.3154 1.05435 10.0517 1.31798L1.38079 9.98822C1.24976 10.1184 1.14587 10.2734 1.07514 10.444C1.00442 10.6147 0.968255 10.7977 0.968755 10.9824V14.1247C0.968755 14.4977 1.11691 14.8553 1.38064 15.1191C1.64436 15.3828 2.00204 15.5309 2.37501 15.5309H14.1875C14.4113 15.5309 14.6259 15.4421 14.7841 15.2838C14.9424 15.1256 15.0313 14.911 15.0313 14.6872C15.0313 14.4634 14.9424 14.2488 14.7841 14.0906C14.6259 13.9323 14.4113 13.8434 14.1875 13.8434H7.78907L15.1824 6.44868C15.3131 6.31809 15.4167 6.16304 15.4874 5.99239C15.5581 5.82174 15.5945 5.63883 15.5945 5.45411C15.5945 5.26939 15.5581 5.08648 15.4874 4.91583C15.4167 4.74518 15.3131 4.59013 15.1824 4.45954ZM5.39844 13.8434H2.65626V11.1013L8.56251 5.19501L11.3047 7.9372L5.39844 13.8434ZM12.5 6.74189L9.75782 3.9997L11.0473 2.71017L13.7895 5.45236L12.5 6.74189Z"
                    fill="#2D2C2C"
                  />
                </svg>
                <span>Edit</span>
              </button>
              <button class="delete-button">
                <svg
                  width="12"
                  height="13"
                  viewBox="0 0 16 17"
                  fill="none"
                  xmlns="http://www.w3.org/2000/svg"
                >
                  <path
                    d="M14.1875 3.375H11.6562V2.53125C11.6562 2.00911 11.4488 1.50835 11.0796 1.13913C10.7104 0.769921 10.2096 0.5625 9.6875 0.5625H6.3125C5.79036 0.5625 5.2896 0.769921 4.92038 1.13913C4.55117 1.50835 4.34375 2.00911 4.34375 2.53125V3.375H1.8125C1.58872 3.375 1.37411 3.46389 1.21588 3.62213C1.05764 3.78036 0.96875 3.99497 0.96875 4.21875C0.96875 4.44253 1.05764 4.65714 1.21588 4.81537C1.37411 4.97361 1.58872 5.0625 1.8125 5.0625H2.09375V14.625C2.09375 14.998 2.24191 15.3556 2.50563 15.6194C2.76935 15.8831 3.12704 16.0312 3.5 16.0312H12.5C12.873 16.0312 13.2306 15.8831 13.4944 15.6194C13.7581 15.3556 13.9062 14.998 13.9062 14.625V5.0625H14.1875C14.4113 5.0625 14.6259 4.97361 14.7841 4.81537C14.9424 4.65714 15.0312 4.44253 15.0312 4.21875C15.0312 3.99497 14.9424 3.78036 14.7841 3.62213C14.6259 3.46389 14.4113 3.375 14.1875 3.375ZM6.03125 2.53125C6.03125 2.45666 6.06088 2.38512 6.11363 2.33238C6.16637 2.27963 6.23791 2.25 6.3125 2.25H9.6875C9.76209 2.25 9.83363 2.27963 9.88637 2.33238C9.93912 2.38512 9.96875 2.45666 9.96875 2.53125V3.375H6.03125V2.53125ZM12.2188 14.3438H3.78125V5.0625H12.2188V14.3438ZM7.15625 7.3125V11.8125C7.15625 12.0363 7.06736 12.2509 6.90912 12.4091C6.75089 12.5674 6.53628 12.6562 6.3125 12.6562C6.08872 12.6562 5.87411 12.5674 5.71588 12.4091C5.55764 12.2509 5.46875 12.0363 5.46875 11.8125V7.3125C5.46875 7.08872 5.55764 6.87411 5.71588 6.71588C5.87411 6.55764 6.08872 6.46875 6.3125 6.46875C6.53628 6.46875 6.75089 6.55764 6.90912 6.71588C7.06736 6.87411 7.15625 7.08872 7.15625 7.3125ZM10.5312 7.3125V11.8125C10.5312 12.0363 10.4424 12.2509 10.2841 12.4091C10.1259 12.5674 9.91128 12.6562 9.6875 12.6562C9.46372 12.6562 9.24911 12.5674 9.09088 12.4091C8.93265 12.2509 8.84375 12.0363 8.84375 11.8125V7.3125C8.84375 7.08872 8.93265 6.87411 9.09088 6.71588C9.24911 6.55764 9.46372 6.46875 9.6875 6.46875C9.91128 6.46875 10.1259 6.55764 10.2841 6.71588C10.4424 6.87411 10.5312 7.08872 10.5312 7.3125Z"
                    fill="white"
                  />
                </svg>
              </button>
            </p>
            <button class="toggle-button ${
              item.status === "enabled" ? "enabled" : "disabled"
            }">
              <span class="button-text">${
                item.status === "enabled" ? "Enabled" : "Disabled"
              }</span>
              <img
                src="${
                  item.status === "enabled"
                    ? "./assets/icons/toggle-on.svg"
                    : "./assets/icons/toggle-off.svg"
                }"
                alt="toggle-button"
              />
            </button>
          </div>
          <div class="image">
            <img src="${item.image}" alt="food" />
            <span class="count">${index + 1}</span>
          </div>
      `;

      dishesBody.appendChild(card);

      // Toggle Dish Card
      const cardToggleButton = card.querySelector(".toggle-button");
      cardToggleButton.addEventListener("click", async () => {
        if (item.status == "enabled") {
          category.dishes[index].status = "disabled";
        } else if (item.status == "disabled") {
          category.dishes[index].status = "enabled";
        }
        try {
          let updatedData = {
            categories: menuData,
          };
          await updateIfHasValue(
            "menus",
            menuData.id,
            "businessID",
            restaurantID,
            updatedData
          );
          populateMenuCategoryTable(menuData.categories);
          populateCardsByCategory(menuData.categories);
        } catch (error) {
          createAlert(
            "couldn't edit category",
            "./assets/icons/bin.svg",
            "red-5"
          );
        }
      });

      const deleteButton = card.querySelector(".delete-button");
      deleteButton.addEventListener("click", () => {
        const modal = existingModal.cloneNode(true);
        modal.id = "delete-dish-modal";
        modal.classList.add("confirm-modal", "lined-modal");

        modal.innerHTML = `
          <div class="modal-heading">
            <p>
              <img src="./assets/icons/bin.svg" alt="bin" />
              <span>Delete?</span>
            </p>
          </div>
          <div class="modal-body">
          <span>Oh no!</span>
            <h2>Would you like to delete this item?</h2>
            <p class="button-section">
              <button id="confirm-delete">
                Delete
              </button>
              <button id="cancel-button">Cancel</button>
            </p>
          </div>
        `;
        main.appendChild(modal);

        modal.showModal();

        const confirmDeleteButton = modal.querySelector("#confirm-delete");
        confirmDeleteButton.addEventListener("click", async () => {
          category.dishes = category.dishes.filter(
            (dish) => dish.id !== item.id
          );
          try {
            let updatedData = {
              categories: menuData,
            };
            await updateIfHasValue(
              "menus",
              menuData.id,
              "businessID",
              restaurantID,
              updatedData
            );
            console.log(menuData);
            createAlert("deleted", "./assets/icons/bin.svg", "red-5");
            setTimeout(() => {
              modal.close();
              modal.remove();
              setTimeout(() => {
                populateMenuCategoryTable(menuData.categories);
                populateCardsByCategory(menuData.categories);
              }, 500);
            }, 800);
          } catch (error) {
            createAlert(
              "couldn't delete category",
              "./assets/icons/bin.svg",
              "red-5"
            );
          }
        });

        const confirmCancelButton = modal.querySelector("#cancel-button");

        confirmCancelButton.addEventListener("click", () => {
          modal.classList.remove("active");
          modal.classList.add("closing");

          setTimeout(() => {
            modal.classList.remove("closing");
            modal.close();
            modal.remove();
          }, 400);
        });
      });

      const editDishButton = card.querySelector(".edit-button");
      editDishButton.addEventListener("click", () => {
        populateForm(2, editDishForm, item);
      });
    });
  });
}

function populateUncategorizedContainer(data) {
  let container = uncategorizedDishes.querySelector(
    "#dishes .uncategorized-container .uncategorized-content"
  );
  container
    .querySelectorAll(":not(template)")
    .forEach((child) => child.remove());

  data.forEach((item, index) => {
    const clone = uncategorizedCardTemplate.content.cloneNode(true);
    const titleSpan = clone.querySelector(".title");
    const priceSpan = clone.querySelector(".price");
    const image = clone.querySelector("img");

    titleSpan.textContent = item.name;
    priceSpan.textContent = `$${item.price.regularPrice}`;
    image.src = item.image;

    const editButton = clone.querySelector(".edit-button");
    editButton.addEventListener("click", () => {
      populateForm(2, editDishForm, item);
      previousPage = "uncategorized";
    });

    const deleteButton = clone.querySelector(".delete-button");
    deleteButton.addEventListener("click", () => {
      const modal = existingModal.cloneNode(true);
      modal.id = "delete-dish-modal";
      modal.classList.add("confirm-modal", "lined-modal");

      modal.innerHTML = `
            <div class="modal-heading">
              <p>
                <img src="./assets/icons/bin.svg" alt="bin" />
                <span>Delete?</span>
              </p>
            </div>
            <div class="modal-body">
            <span>Oh no!</span>
              <h2>Would you like to delete this item?</h2>
              <p class="button-section">
                <button id="confirm-delete">
                  Delete
                </button>
                <button id="cancel-button">Cancel</button>
              </p>
            </div>
          `;
      main.appendChild(modal);

      modal.showModal();

      const confirmDeleteButton = modal.querySelector("#confirm-delete");

      confirmDeleteButton.addEventListener("click", async () => {
        data = data.filter((dish) => dish.id !== item.id);
        try {
          let updatedData = {
            uncategorizedItems: data,
          };
          let response = await updateIfHasValue(
            "menus",
            menuData.id,
            "businessID",
            restaurantID,
            updatedData
          );
          menuData.uncategorizedItems = response.uncategorizedItems;
          createAlert("deleted", "./assets/icons/bin.svg", "red-5");
          setTimeout(() => {
            modal.close();
            modal.remove();
            setTimeout(() => {
              populateUncategorizedContainer(menuData.uncategorizedItems);
            }, 500);
          }, 800);
        } catch (error) {
          createAlert(
            "couldn't edit category",
            "./assets/icons/bin.svg",
            "red-5"
          );
        }
      });

      const confirmCancelButton = modal.querySelector("#cancel-button");

      confirmCancelButton.addEventListener("click", () => {
        modal.classList.remove("active");
        modal.classList.add("closing");

        setTimeout(() => {
          modal.classList.remove("closing");
          modal.close();
          modal.remove();
        }, 400);
      });
    });

    container.appendChild(clone);
  });
}

const populateDishesPage = async () => {
  let response = await getFiltered("menus", {
    businessID: restaurantID,
  });

  menuData = response[0];

  // populate infocards
  populateDishesInfoCard();

  // populate menu table
  populateMenuCategoryTable(menuData.categories);

  // populate dishes page
  populateCardsByCategory(menuData.categories);

  // populate uncategorized container
  populateUncategorizedContainer(menuData.uncategorizedItems);
};

const populateDeliveryPartnerInfoCards = async () => {
  let availablePartners = deliveryPartnersData.filter((item) =>
    Object.entries({ status: "available" }).every(
      ([key, value]) => item[key] === value
    )
  );
  let unavailablePartners = deliveryPartnersData.filter((item) =>
    Object.entries({ status: "unavailable" }).every(
      ([key, value]) => item[key] === value
    )
  );

  const availablePartnersNo = document.querySelector(
    "#delivery  header .info-card > div > p:nth-child(1) > span:nth-child(2)"
  );
  const unavailablePartnersNo = document.querySelector(
    "#delivery  header .info-card > div > p:nth-child(2) > span:nth-child(2)"
  );

  availablePartnersNo.textContent = availablePartners.length;
  unavailablePartnersNo.textContent = unavailablePartners.length;
};

const populateFormFields = (formContainer, data, id) => {
  const days = [
    "monday",
    "tuesday",
    "wednesday",
    "thursday",
    "friday",
    "saturday",
    "sunday",
  ];

  days.forEach((day, index) => {
    const dayData = data.find((item) => item.day.toLowerCase() === day);
    if (dayData) {
      const dayLower = day.toLowerCase();
      const timeDropdown = formContainer.querySelector(
        `.form-row:nth-child(${index + 1}) .dropdown button`
      );
      timeDropdown.textContent = dayData.time;
      formContainer
        .querySelector(`.form-row:nth-child(${index + 1}) .dropdown`)
        .classList.add("selected");

      const percentageInput = formContainer.querySelector(
        `#${dayLower}-percentage-discount${id}`
      );
      percentageInput.value = dayData.discountPercentage;

      const minOrderInput = formContainer.querySelector(
        `#${dayLower}-min-order${id}`
      );
      minOrderInput.value = dayData.minOrder;

      formContainer
        .querySelector(`.form-row:nth-child(${index + 1})`)
        .classList.add("active");
    }
  });
};

const populateDiscountPages = async () => {
  let data = restaurantData.discount;

  // populate daily delivery discount
  const dailyDeliveryDiscount = document.getElementById("delivery-form");
  const dailyDeliveryFormContainer =
    dailyDeliveryDiscount.querySelector(".form-group");
  populateFormFields(
    dailyDeliveryFormContainer,
    data.dailyDiscount.deliveryOrderDiscount,
    1
  );

  const dailyDineinDiscount = document.getElementById("dinein-form");
  const dailyDineinFormContainer =
    dailyDineinDiscount.querySelector(".form-group");
  populateFormFields(
    dailyDineinFormContainer,
    data.dailyDiscount.dineinOrderDiscount,
    2
  );

  const dailyTakeawayDiscount = document.getElementById("takeaway-form");
  const dailyTakeawayFormContainer =
    dailyTakeawayDiscount.querySelector(".form-group");
  populateFormFields(
    dailyTakeawayFormContainer,
    data.dailyDiscount.takeawayOrderDiscount,
    3
  );

  const deliveryFeeDiscount = document.getElementById("delivery-fee-form");
  const deliveryFeeFormContainer =
    deliveryFeeDiscount.querySelector(".form-group");
  populateFormFields(deliveryFeeFormContainer, data.deliveryFeeDiscount, 4);
};

// Function to populate the table
const populateFreelanceDeliveryTable = (data) => {
  const sortedData = [...data].sort((a, b) => b.pinned - a.pinned);
  deliveryPartnerTable.innerHTML = "";

  const createPinButton = () => {
    return `
      <button class="pin">
        <svg class="inactive" width="15" height="15" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M13.8844 4.90911L10.0913 1.11661C9.97516 1.00039 9.8373 0.908189 9.68555 0.845284C9.5338 0.782378 9.37115 0.75 9.20688 0.75C9.04261 0.75 8.87995 0.782378 8.7282 0.845284C8.57645 0.908189 8.43859 1.00039 8.3225 1.11661L5.075 4.37536C4.29813 4.16723 2.7825 4.01786 1.21625 5.28161C1.07948 5.39165 0.967415 5.52927 0.887361 5.6855C0.807307 5.84172 0.761057 6.01306 0.751623 6.18836C0.742189 6.36365 0.769783 6.53896 0.832608 6.70288C0.895432 6.8668 0.992078 7.01565 1.11625 7.13973L3.9575 9.98161L1.47 12.4691C1.32911 12.61 1.24995 12.8011 1.24995 13.0004C1.24995 13.1996 1.32911 13.3907 1.47 13.5316C1.6109 13.6725 1.80199 13.7517 2.00125 13.7517C2.20051 13.7517 2.39161 13.6725 2.5325 13.5316L5.02 11.0435L7.86188 13.8847C8.09604 14.1186 8.41341 14.2501 8.74438 14.2504C8.77313 14.2504 8.8025 14.2504 8.83188 14.2472C9.01028 14.2343 9.18383 14.1833 9.34082 14.0976C9.49781 14.0119 9.63457 13.8935 9.74188 13.7504C10.035 13.361 10.4294 12.7422 10.6444 12.0004C10.8594 11.2585 10.86 10.5704 10.6544 9.91286L13.8856 6.67098C14.1181 6.43664 14.2484 6.11987 14.2482 5.78979C14.2479 5.45971 14.1172 5.14312 13.8844 4.90911ZM9.24313 9.20848C9.13216 9.31986 9.05901 9.4633 9.034 9.61852C9.00899 9.77374 9.03339 9.93289 9.10375 10.0735C9.31813 10.5022 9.535 11.386 8.71125 12.6141L2.38 6.28348C3.69813 5.37161 4.85063 5.86848 4.9425 5.90848C5.0825 5.98117 5.24229 6.00651 5.39791 5.98071C5.55353 5.9549 5.69658 5.87934 5.80563 5.76536L9.2075 2.35223L12.645 5.78973L9.24313 9.20848Z" fill="#2D2C2C" fill-opacity="0.4"/>
        </svg>
        <svg class="active" width="13" height="13" viewBox="0 0 13 13" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M12.7082 5.49996L9.3663 8.85308C9.6513 9.64496 9.76942 10.9712 8.54192 12.6031C8.45607 12.7178 8.34646 12.8127 8.22058 12.8812C8.0947 12.9497 7.95552 12.9902 7.81255 13C7.7888 13 7.76567 13 7.74192 13C7.61052 13 7.4804 12.9741 7.359 12.9238C7.2376 12.8735 7.12731 12.7998 7.03442 12.7068L4.01817 9.68746L1.3538 12.3537C1.25998 12.4475 1.13273 12.5002 1.00005 12.5002C0.867364 12.5002 0.740116 12.4475 0.646295 12.3537C0.552475 12.2599 0.499767 12.1326 0.499767 12C0.499767 11.8673 0.552475 11.74 0.646295 11.6462L3.31255 8.98183L0.293796 5.96308C0.19457 5.86373 0.117379 5.7446 0.067248 5.61344C0.0171169 5.48228 -0.00483088 5.34203 0.00283378 5.20183C0.0104984 5.06162 0.0476035 4.92461 0.111734 4.79969C0.175864 4.67478 0.265581 4.56477 0.375045 4.47683C1.9638 3.19496 3.48442 3.44683 4.15005 3.65558L7.50005 0.293706C7.59291 0.200824 7.70316 0.127144 7.8245 0.0768747C7.94584 0.0266058 8.07589 0.000732422 8.20723 0.000732422C8.33857 0.000732422 8.46863 0.0266058 8.58997 0.0768747C8.71131 0.127144 8.82156 0.200824 8.91442 0.293706L12.7075 4.08621C12.895 4.27364 13.0004 4.52785 13.0005 4.79295C13.0006 5.05805 12.8955 5.31236 12.7082 5.49996Z" fill="#FF0828"/>
        </svg>
      </button>
    `;
  };

  sortedData.forEach((partner, index) => {
    const row = document.createElement("tr");
    row.setAttribute("data-row-id", `partner-row-${index + 1}`);

    row.innerHTML = `
      <td>
        <label>
          <input type="checkbox" />
          <svg
            width="20"
            height="20"
            viewBox="0 0 20 20"
            fill="none"
            xmlns="http://www.w3.org/2000/svg"
          >
            <path
              d="M10 0.25C8.07164 0.25 6.18657 0.821828 4.58319 1.89317C2.97982 2.96451 1.73013 4.48726 0.992179 6.26884C0.254225 8.05042 0.061142 10.0108 0.437348 11.9021C0.813554 13.7934 1.74215 15.5307 3.10571 16.8943C4.46928 18.2579 6.20656 19.1865 8.09787 19.5627C9.98919 19.9389 11.9496 19.7458 13.7312 19.0078C15.5127 18.2699 17.0355 17.0202 18.1068 15.4168C19.1782 13.8134 19.75 11.9284 19.75 10C19.7473 7.41498 18.7192 4.93661 16.8913 3.10872C15.0634 1.28084 12.585 0.25273 10 0.25ZM14.2806 8.28063L9.03063 13.5306C8.96097 13.6004 8.87826 13.6557 8.78721 13.6934C8.69616 13.7312 8.59857 13.7506 8.5 13.7506C8.40144 13.7506 8.30385 13.7312 8.2128 13.6934C8.12175 13.6557 8.03903 13.6004 7.96938 13.5306L5.71938 11.2806C5.57865 11.1399 5.49959 10.949 5.49959 10.75C5.49959 10.551 5.57865 10.3601 5.71938 10.2194C5.86011 10.0786 6.05098 9.99958 6.25 9.99958C6.44903 9.99958 6.6399 10.0786 6.78063 10.2194L8.5 11.9397L13.2194 7.21937C13.2891 7.14969 13.3718 7.09442 13.4628 7.0567C13.5539 7.01899 13.6515 6.99958 13.75 6.99958C13.8486 6.99958 13.9461 7.01899 14.0372 7.0567C14.1282 7.09442 14.2109 7.14969 14.2806 7.21937C14.3503 7.28906 14.4056 7.37178 14.4433 7.46283C14.481 7.55387 14.5004 7.65145 14.5004 7.75C14.5004 7.84855 14.481 7.94613 14.4433 8.03717C14.4056 8.12822 14.3503 8.21094 14.2806 8.28063Z"
              fill="none"
              fill-opacity="0.6"
            />
          </svg>
        </label>
      </td>
      <td style="display: none">${partner.id}</td>
      <td>
        <img src="${partner.avatar}" alt="avatar" width="35" />
        <div>
          <span>${partner.name}</span>
          <span>${partner.phone}</span>
          <p>
            <img src="./assets/icons/star.svg" alt="star"/>
            <span>${partner.reviews.averageRating}</span>
            <span>${partner.distance}</span>
          </p>
        </div>
      </td>
      <td>${partner.phone}</td>
      <td>
        <img src="./assets/icons/star.svg" alt="star"/>
        <span>${partner.reviews.averageRating}</span>
      </td>
      <td>${partner.averageDeliveriesPerDay}</td>
      <td>${partner.averageDeliveries}</td>
      <td>${partner.otherDeliveries}</td>
      <td>${formatDate(partner.lastDelivery)} ${formatTime(
      partner.lastDelivery
    )}</td>
      <td>${partner.distance}</td>
      <td class="status"><span>${partner.status}</span></td>
      <td>${createActionDropdown(partner)}</td>
      <td>${createPinButton()}</td>
    `;

    deliveryPartnerTable.appendChild(row);
    attachRowEventListeners(row, partner);
  });

  function createActionDropdown(partner) {
    return `
      <div class="static-dropdown">
        <button class="dropdown-button">
          <svg
            width="16"
            height="6"
            viewBox="0 0 22 6"
            fill="none"
            xmlns="http://www.w3.org/2000/svg"
          >
            <path
              d="M13.625 3C13.625 3.51918 13.471 4.02669 13.1826 4.45837C12.8942 4.89005 12.4842 5.2265 12.0045 5.42518C11.5249 5.62386 10.9971 5.67585 10.4879 5.57456C9.97869 5.47328 9.51096 5.22327 9.14385 4.85616C8.77673 4.48904 8.52673 4.02131 8.42544 3.51211C8.32415 3.00291 8.37614 2.47511 8.57482 1.99546C8.7735 1.5158 9.10995 1.10583 9.54163 0.817392C9.97331 0.528953 10.4808 0.375 11 0.375C11.6962 0.375 12.3639 0.651562 12.8562 1.14384C13.3484 1.63613 13.625 2.30381 13.625 3ZM3.5 0.375C2.98083 0.375 2.47331 0.528953 2.04163 0.817392C1.60995 1.10583 1.2735 1.5158 1.07482 1.99546C0.876137 2.47511 0.824154 3.00291 0.92544 3.51211C1.02673 4.02131 1.27673 4.48904 1.64385 4.85616C2.01096 5.22327 2.47869 5.47328 2.98789 5.57456C3.49709 5.67585 4.02489 5.62386 4.50455 5.42518C4.9842 5.2265 5.39417 4.89005 5.68261 4.45837C5.97105 4.02669 6.125 3.51918 6.125 3C6.125 2.30381 5.84844 1.63613 5.35616 1.14384C4.86387 0.651562 4.19619 0.375 3.5 0.375ZM18.5 0.375C17.9808 0.375 17.4733 0.528953 17.0416 0.817392C16.61 1.10583 16.2735 1.5158 16.0748 1.99546C15.8761 2.47511 15.8242 3.00291 15.9254 3.51211C16.0267 4.02131 16.2767 4.48904 16.6438 4.85616C17.011 5.22327 17.4787 5.47328 17.9879 5.57456C18.4971 5.67585 19.0249 5.62386 19.5045 5.42518C19.9842 5.2265 20.3942 4.89005 20.6826 4.45837C20.971 4.02669 21.125 3.51918 21.125 3C21.125 2.65528 21.0571 2.31394 20.9252 1.99546C20.7933 1.67698 20.5999 1.3876 20.3562 1.14384C20.1124 0.900091 19.823 0.706735 19.5045 0.574816C19.1861 0.442897 18.8447 0.375 18.5 0.375Z"
              fill="#FF0828"
            />
          </svg>  
        </button>
        <div class="dropdown">
          <div class="dropdown-heading"><span>Action</span></div>
          <ul>
            <li class="${partner.pinned === true ? "unpin" : "pin"} pin-button">
            <svg width="18" height="18" viewBox="0 0 21 21" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M20.3265 6.36269L14.6368 0.673936C14.4627 0.499605 14.2559 0.361307 14.0283 0.266949C13.8007 0.172591 13.5567 0.124023 13.3103 0.124023C13.0639 0.124023 12.8199 0.172591 12.5923 0.266949C12.3647 0.361307 12.1579 0.499605 11.9837 0.673936L7.11247 5.56206C5.94716 5.24987 3.67372 5.02581 1.32435 6.92144C1.11919 7.0865 0.951092 7.29292 0.831011 7.52727C0.71093 7.76161 0.641555 8.01862 0.627404 8.28156C0.613254 8.5445 0.654644 8.80747 0.748881 9.05334C0.843117 9.29922 0.988086 9.5225 1.17435 9.70862L5.43622 13.9714L1.70497 17.7027C1.49363 17.914 1.3749 18.2007 1.3749 18.4996C1.3749 18.7984 1.49363 19.0851 1.70497 19.2964C1.91632 19.5078 2.20296 19.6265 2.50185 19.6265C2.80073 19.6265 3.08738 19.5078 3.29872 19.2964L7.02997 15.5642L11.2928 19.8261C11.644 20.177 12.1201 20.3742 12.6165 20.3746C12.6597 20.3746 12.7037 20.3746 12.7478 20.3699C13.0154 20.3505 13.2757 20.274 13.5112 20.1454C13.7467 20.0168 13.9518 19.8392 14.1128 19.6246C14.5525 19.0405 15.144 18.1124 15.4665 16.9996C15.789 15.8867 15.79 14.8546 15.4815 13.8683L20.3284 9.0055C20.6771 8.65399 20.8726 8.17883 20.8722 7.68371C20.8719 7.18859 20.6757 6.7137 20.3265 6.36269ZM13.3647 12.8117C13.1982 12.9788 13.0885 13.194 13.051 13.4268C13.0135 13.6596 13.0501 13.8984 13.1556 14.1092C13.4772 14.7524 13.8025 16.078 12.5668 17.9202L3.06997 8.42425C5.04716 7.05644 6.77591 7.80175 6.91372 7.86175C7.12372 7.97078 7.36341 8.00879 7.59683 7.97008C7.83026 7.93137 8.04484 7.81803 8.20841 7.64706L13.3112 2.52737L18.4675 7.68362L13.3647 12.8117Z" fill="#2D2C2C" fill-opacity="0.2"/>
            </svg>
            <span>${partner.pinned === true ? "Unpin" : "Pin to the top"}</span>
          </li>
            <li class="${partner.blockStatus ? "block" : "unblock"}">
              <svg width="17" height="18" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M19.375 3.24971V8.76221C19.3859 10.2557 19.1395 11.7399 18.6466 13.1497C18.5968 13.2888 18.5202 13.4168 18.421 13.5263C18.3218 13.6358 18.202 13.7247 18.0685 13.7879C17.9349 13.8511 17.7902 13.8873 17.6427 13.8946C17.4951 13.9019 17.3476 13.8801 17.2084 13.8303C17.0693 13.7806 16.9414 13.7039 16.8319 13.6048C16.7224 13.5056 16.6335 13.3858 16.5703 13.2522C16.5071 13.1187 16.4708 12.974 16.4635 12.8264C16.4562 12.6789 16.4781 12.5313 16.5278 12.3922C16.9341 11.2255 17.1361 9.99754 17.125 8.76221V3.62471H8.24969C7.95132 3.62471 7.66517 3.50618 7.45419 3.29521C7.24321 3.08423 7.12469 2.79808 7.12469 2.49971C7.12469 2.20134 7.24321 1.91519 7.45419 1.70422C7.66517 1.49324 7.95132 1.37471 8.24969 1.37471H17.5C17.9973 1.37471 18.4742 1.57225 18.8258 1.92389C19.1775 2.27552 19.375 2.75243 19.375 3.24971ZM18.3325 17.4931C18.4319 17.6025 18.5088 17.7303 18.5588 17.8694C18.6088 18.0084 18.631 18.1559 18.624 18.3035C18.617 18.4511 18.5809 18.5959 18.518 18.7296C18.455 18.8633 18.3664 18.9833 18.257 19.0827C18.1477 19.1821 18.0199 19.259 17.8808 19.309C17.7418 19.359 17.5942 19.3812 17.4466 19.3741C17.299 19.3671 17.1543 19.3311 17.0206 19.2682C16.8869 19.2052 16.7669 19.1165 16.6675 19.0072L15.5669 17.796C14.1291 19.1229 12.4332 20.139 10.585 20.781C10.2051 20.9084 9.79397 20.9084 9.41406 20.781C7.94688 20.2916 0.625 17.421 0.625 8.76221V3.24971C0.625 2.95863 0.692772 2.67154 0.822949 2.41119C0.953126 2.15083 1.14213 1.92436 1.375 1.74971C1.37466 1.52261 1.44307 1.30073 1.57121 1.11324C1.69935 0.925755 1.88124 0.781437 2.09295 0.699269C2.30466 0.6171 2.53628 0.600928 2.75736 0.65288C2.97843 0.704831 3.17861 0.822473 3.33156 0.990336L18.3325 17.4931ZM14.0538 16.1319L2.875 3.83471V8.76221C2.875 15.6585 8.54969 18.0931 10 18.606C11.5038 18.0621 12.8825 17.2207 14.0538 16.1319Z" fill="#2D2C2C" fill-opacity="0.2"/>
              </svg>
              <span>${partner.blockStatus ? "Block" : "Unblock"}</span>
            </li>
            <li class="remove-button">
              <svg width="19" height="19" viewBox="0 0 22 22" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M15.875 11C15.875 11.2984 15.7565 11.5845 15.5455 11.7955C15.3345 12.0065 15.0484 12.125 14.75 12.125H7.25C6.95164 12.125 6.66549 12.0065 6.45451 11.7955C6.24353 11.5845 6.125 11.2984 6.125 11C6.125 10.7016 6.24353 10.4155 6.45451 10.2045C6.66549 9.99353 6.95164 9.875 7.25 9.875H14.75C15.0484 9.875 15.3345 9.99353 15.5455 10.2045C15.7565 10.4155 15.875 10.7016 15.875 11ZM21.125 11C21.125 13.0025 20.5312 14.9601 19.4186 16.6251C18.3061 18.2902 16.7248 19.5879 14.8747 20.3543C13.0246 21.1206 10.9888 21.3211 9.02471 20.9305C7.06066 20.5398 5.25656 19.5755 3.84055 18.1595C2.42454 16.7435 1.46023 14.9393 1.06955 12.9753C0.678878 11.0112 0.879387 8.97543 1.64572 7.12533C2.41206 5.27523 3.70981 3.69392 5.37486 2.58137C7.0399 1.46882 8.99747 0.875 11 0.875C13.6844 0.877978 16.258 1.94567 18.1562 3.84383C20.0543 5.74199 21.122 8.3156 21.125 11ZM18.875 11C18.875 9.44247 18.4131 7.91992 17.5478 6.62488C16.6825 5.32985 15.4526 4.32049 14.0136 3.72445C12.5747 3.12841 10.9913 2.97246 9.46367 3.27632C7.93607 3.58017 6.53288 4.3302 5.43154 5.43153C4.3302 6.53287 3.58018 7.93606 3.27632 9.46366C2.97246 10.9913 3.12841 12.5747 3.72445 14.0136C4.32049 15.4526 5.32985 16.6825 6.62489 17.5478C7.91993 18.4131 9.44248 18.875 11 18.875C13.0879 18.8728 15.0896 18.0424 16.566 16.566C18.0424 15.0896 18.8728 13.0879 18.875 11Z" fill="#FF0828"/>
              </svg>
              <span>Remove</span>
            </li>
          </ul>
        </div>
      </div>
    `;
  }

  function attachRowEventListeners(row, partner) {
    const deliveryPartnerName = row.querySelector(
      "td:nth-child(3) > div > span:nth-child(1)"
    );
    deliveryPartnerName.addEventListener("click", () =>
      populateDeliveryPartnerModal(partner.id)
    );

    const blockModal = document.getElementById("block-modal");
    const blockButton =
      row.querySelector(".block") || row.querySelector(".unblock");
    let isBlocked = false;

    const setBlock = () => {
      isBlocked = restaurantData.blocked.blockedPartners.some(
        (item) => item.id == partner.id
      );
      blockModal.classList.toggle("block", !isBlocked);
      blockModal.classList.toggle("unblock", isBlocked);
      blockButton.classList.toggle("block", !isBlocked);
      blockButton.classList.toggle("unblock", isBlocked);
      blockButton.querySelector("span").textContent = isBlocked
        ? "Unblock"
        : "Block";
    };

    setBlock();

    blockButton.addEventListener("click", () => {
      toggleModalContent(isBlocked);
      blockModal.showModal();
    });

    function toggleModalContent(isBlocked) {
      blockModal.style.zIndex = 50;
      blockModal.innerHTML = isBlocked
        ? createUnblockModalContent()
        : createBlockModalContent();
      bindModalEventListeners(blockModal, partner);
    }

    function createUnblockModalContent() {
      return `
        <div class="modal-heading">
          <p>
            <img src="./assets/icons/smiles.svg" alt="smile" />
            <span>Oh Yeah! 👍🏾</span>
          </p>
          <button type="button" class="close-button">
            <img src="./assets/icons/close.svg" alt="close icon" class="close-icon" />
          </button>
        </div>
        <div class="modal-body">
          <h2>Great! <br/> Jackson will be able to run deliveries for your restaurant, when you unblock Delivery Partner.</h2>
          <p class="button-section">
            <button id="unblock-confirm-button" class="open-modal-btn">
              <span> Unblock Delivery Partner </span>
              <img src="./assets/icons/double-check.svg" alt="check-icon"/>
            </button>
            <button id="cancel-button">Cancel</button>
          </p>
        </div>
      `;
    }

    function createBlockModalContent() {
      return `
        <div class="modal-heading">
          <p>
            <img src="./assets/icons/frown.svg" alt="frown" />
            <span>Oh No!</span>
          </p>
          <button type="button" class="close-button">
            <img src="./assets/icons/close.svg" alt="close icon" class="close-icon" />
          </button>
        </div>
        <div class="modal-body">
          <h2>Do you want to block this Delivery Partner?</h2>
          <p class="button-section">
            <button id="block-confirm-button" class="open-modal-btn">
              <span> Block Delivery Partner </span>
              <img src="./assets/icons/block-icon.svg" alt="block-icon"/>
            </button>
            <button id="cancel-button">Cancel</button>
          </p>
        </div>
      `;
    }

    function bindModalEventListeners(modal, partner) {
      modal
        .querySelectorAll(".close-button, #cancel-button")
        .forEach((button) =>
          button.addEventListener("click", () => modal.close())
        );

      modal
        .querySelector("#block-confirm-button")
        ?.addEventListener("click", async () => {
          modal.close();
          await handleBlock(partner);
        });

      modal
        .querySelector("#unblock-confirm-button")
        ?.addEventListener("click", async () => {
          modal.close();
          await handleUnblock(partner);
        });
    }

    async function handleBlock(partner) {
      try {
        restaurantData.blocked.blockedPartners.push({
          id: partner.id,
          name: partner.name,
        });
        await update("restaurants", restaurantID, {
          blocked: restaurantData.blocked,
        });
        setBlock();
      } catch (error) {
        createAlert(
          "Couldn't block delivery partner",
          "./assets/icons/bin.svg",
          "red-5"
        );
      }
    }

    async function handleUnblock(partner) {
      try {
        restaurantData.blocked.blockedPartners =
          restaurantData.blocked.blockedPartners.filter(
            (p) => p.id !== partner.id
          );
        await update("restaurants", restaurantID, {
          blocked: restaurantData.blocked,
        });
        setBlock();
      } catch (error) {
        createAlert(
          "Couldn't unblock delivery partner",
          "./assets/icons/bin.svg",
          "red-5"
        );
      }
    }

    const pinButton = row.querySelector(".pin-button");
    pinButton.addEventListener("click", () => togglePin(partner.id));

    const removeButton = row.querySelector(".remove-button");
    removeButton.addEventListener("click", async () => {
      try {
        deliveryPartnersData = deliveryPartnersData.filter(
          (item) => item.id !== partner.id
        );
        data = data.filter((item) => item.id !== partner.id);
        await remove("delivery-partners", partner.id);
        populateDeliveryPartnerPage("freelance");
      } catch (error) {
        console.error("Failed to remove delivery partner:", error);
      }
    });
  }

  async function togglePin(rowId) {
    const rowData = deliveryPartnersData.find((row) => row.id === rowId);
    if (rowData) {
      rowData.pinned = !rowData.pinned;
      try {
        await update("delivery-partners", rowId, { pinned: rowData.pinned });
        populateDeliveryPartnerPage("freelance");
      } catch (error) {
        console.error("Failed to update pinned status:", error);
      }
    }
  }
};

const populateInhouseDeliveryTable = (data) => {
  const inhouseDeliveryTable = document.querySelector("#inhouse-table tbody");

  const updateDataWithFilteredResults = async (data) => {
    const updatePromises = data.map(async (item) => {
      try {
        const response = await getFiltered("delivery-partners", {
          email: item.email,
        });
        return response[0] || item;
      } catch (error) {
        console.error("Error fetching data for:", item.email, error);
        return item;
      }
    });
    return Promise.all(updatePromises);
  };

  const renderTableRows = (sortedData) => {
    inhouseDeliveryTable.innerHTML = "";
    sortedData.forEach((partner, index) => {
      const row = document.createElement("tr");
      row.setAttribute("data-row-id", `partner-row-${index + 1}`);
      row.innerHTML = createRowHTML(partner);
      inhouseDeliveryTable.appendChild(row);
      attachRowEventListeners(row, partner);
    });
  };

  const createRowHTML = (partner) => `
    <td>
      <label>
        <input type="checkbox" />
        <svg
          width="20"
          height="20"
          viewBox="0 0 20 20"
          fill="none"
          xmlns="http://www.w3.org/2000/svg"
        >
          <path
            d="M10 0.25C8.07164 0.25 6.18657 0.821828 4.58319 1.89317C2.97982 2.96451 1.73013 4.48726 0.992179 6.26884C0.254225 8.05042 0.061142 10.0108 0.437348 11.9021C0.813554 13.7934 1.74215 15.5307 3.10571 16.8943C4.46928 18.2579 6.20656 19.1865 8.09787 19.5627C9.98919 19.9389 11.9496 19.7458 13.7312 19.0078C15.5127 18.2699 17.0355 17.0202 18.1068 15.4168C19.1782 13.8134 19.75 11.9284 19.75 10C19.7473 7.41498 18.7192 4.93661 16.8913 3.10872C15.0634 1.28084 12.585 0.25273 10 0.25ZM14.2806 8.28063L9.03063 13.5306C8.96097 13.6004 8.87826 13.6557 8.78721 13.6934C8.69616 13.7312 8.59857 13.7506 8.5 13.7506C8.40144 13.7506 8.30385 13.7312 8.2128 13.6934C8.12175 13.6557 8.03903 13.6004 7.96938 13.5306L5.71938 11.2806C5.57865 11.1399 5.49959 10.949 5.49959 10.75C5.49959 10.551 5.57865 10.3601 5.71938 10.2194C5.86011 10.0786 6.05098 9.99958 6.25 9.99958C6.44903 9.99958 6.6399 10.0786 6.78063 10.2194L8.5 11.9397L13.2194 7.21937C13.2891 7.14969 13.3718 7.09442 13.4628 7.0567C13.5539 7.01899 13.6515 6.99958 13.75 6.99958C13.8486 6.99958 13.9461 7.01899 14.0372 7.0567C14.1282 7.09442 14.2109 7.14969 14.2806 7.21937C14.3503 7.28906 14.4056 7.37178 14.4433 7.46283C14.481 7.55387 14.5004 7.65145 14.5004 7.75C14.5004 7.84855 14.481 7.94613 14.4433 8.03717C14.4056 8.12822 14.3503 8.21094 14.2806 8.28063Z"
            fill="none"
            fill-opacity="0.6"
          />
        </svg>
      </label>
    </td>
    <td style="display: none">${partner.id || 0}</td>
    <td>
      <img src="${
        partner.avatar || "./assets/icons/user-icon.svg"
      }" alt="avatar" width="35" />
      <div>
        <span>${partner.name || partner.email}</span>
        <span>${partner.phone || "--------"}</span>
        <p>
          <img src="./assets/icons/star.svg" alt="star"/>
          <span>${partner.reviews?.averageRating || "--"}</span>
          <span>${partner.distance || "--"}</span>
        </p>
      </div>
    </td>
    <td>${partner.phone || "--------"}</td>
    <td><img src="./assets/icons/star.svg" alt="star"/><span>${
      partner.reviews?.averageRating || "--"
    }</span></td>
    <td>${partner.averageDeliveriesPerDay || "--"}</td>
    <td>${partner.averageDeliveries || "--"}</td>
    <td>${partner.otherDeliveries || "--"}</td>
    <td>${
      partner.lastDelivery
        ? formatDate(partner.lastDelivery) +
          " " +
          formatTime(partner.lastDelivery)
        : "----"
    }</td>
    <td>${partner.distance || "--"}</td>
    <td class="${partner.status || "pending"} status"><span>${
    partner.status || "pending"
  }</span></td>
    <td>${createActionDropdown(partner)}</td>
    <td>
      <button class="pin">
      <svg class="inactive" width="15" height="15" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M13.8844 4.90911L10.0913 1.11661C9.97516 1.00039 9.8373 0.908189 9.68555 0.845284C9.5338 0.782378 9.37115 0.75 9.20688 0.75C9.04261 0.75 8.87995 0.782378 8.7282 0.845284C8.57645 0.908189 8.43859 1.00039 8.3225 1.11661L5.075 4.37536C4.29813 4.16723 2.7825 4.01786 1.21625 5.28161C1.07948 5.39165 0.967415 5.52927 0.887361 5.6855C0.807307 5.84172 0.761057 6.01306 0.751623 6.18836C0.742189 6.36365 0.769783 6.53896 0.832608 6.70288C0.895432 6.8668 0.992078 7.01565 1.11625 7.13973L3.9575 9.98161L1.47 12.4691C1.32911 12.61 1.24995 12.8011 1.24995 13.0004C1.24995 13.1996 1.32911 13.3907 1.47 13.5316C1.6109 13.6725 1.80199 13.7517 2.00125 13.7517C2.20051 13.7517 2.39161 13.6725 2.5325 13.5316L5.02 11.0435L7.86188 13.8847C8.09604 14.1186 8.41341 14.2501 8.74438 14.2504C8.77313 14.2504 8.8025 14.2504 8.83188 14.2472C9.01028 14.2343 9.18383 14.1833 9.34082 14.0976C9.49781 14.0119 9.63457 13.8935 9.74188 13.7504C10.035 13.361 10.4294 12.7422 10.6444 12.0004C10.8594 11.2585 10.86 10.5704 10.6544 9.91286L13.8856 6.67098C14.1181 6.43664 14.2484 6.11987 14.2482 5.78979C14.2479 5.45971 14.1172 5.14312 13.8844 4.90911ZM9.24313 9.20848C9.13216 9.31986 9.05901 9.4633 9.034 9.61852C9.00899 9.77374 9.03339 9.93289 9.10375 10.0735C9.31813 10.5022 9.535 11.386 8.71125 12.6141L2.38 6.28348C3.69813 5.37161 4.85063 5.86848 4.9425 5.90848C5.0825 5.98117 5.24229 6.00651 5.39791 5.98071C5.55353 5.9549 5.69658 5.87934 5.80563 5.76536L9.2075 2.35223L12.645 5.78973L9.24313 9.20848Z" fill="#2D2C2C" fill-opacity="0.4"/>
      </svg>
      <svg class="active" width="13" height="13" viewBox="0 0 13 13" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M12.7082 5.49996L9.3663 8.85308C9.6513 9.64496 9.76942 10.9712 8.54192 12.6031C8.45607 12.7178 8.34646 12.8127 8.22058 12.8812C8.0947 12.9497 7.95552 12.9902 7.81255 13C7.7888 13 7.76567 13 7.74192 13C7.61052 13 7.4804 12.9741 7.359 12.9238C7.2376 12.8735 7.12731 12.7998 7.03442 12.7068L4.01817 9.68746L1.3538 12.3537C1.25998 12.4475 1.13273 12.5002 1.00005 12.5002C0.867364 12.5002 0.740116 12.4475 0.646295 12.3537C0.552475 12.2599 0.499767 12.1326 0.499767 12C0.499767 11.8673 0.552475 11.74 0.646295 11.6462L3.31255 8.98183L0.293796 5.96308C0.19457 5.86373 0.117379 5.7446 0.067248 5.61344C0.0171169 5.48228 -0.00483088 5.34203 0.00283378 5.20183C0.0104984 5.06162 0.0476035 4.92461 0.111734 4.79969C0.175864 4.67478 0.265581 4.56477 0.375045 4.47683C1.9638 3.19496 3.48442 3.44683 4.15005 3.65558L7.50005 0.293706C7.59291 0.200824 7.70316 0.127144 7.8245 0.0768747C7.94584 0.0266058 8.07589 0.000732422 8.20723 0.000732422C8.33857 0.000732422 8.46863 0.0266058 8.58997 0.0768747C8.71131 0.127144 8.82156 0.200824 8.91442 0.293706L12.7075 4.08621C12.895 4.27364 13.0004 4.52785 13.0005 4.79295C13.0006 5.05805 12.8955 5.31236 12.7082 5.49996Z" fill="#FF0828"/>
      </svg>
    </button>
    </td>
  `;

  const createActionDropdown = (partner) => `
    <div class="static-dropdown">
      <button class="dropdown-button">
        <svg
          width="16"
          height="6"
          viewBox="0 0 22 6"
          fill="none"
          xmlns="http://www.w3.org/2000/svg"
        >
          <path
            d="M13.625 3C13.625 3.51918 13.471 4.02669 13.1826 4.45837C12.8942 4.89005 12.4842 5.2265 12.0045 5.42518C11.5249 5.62386 10.9971 5.67585 10.4879 5.57456C9.97869 5.47328 9.51096 5.22327 9.14385 4.85616C8.77673 4.48904 8.52673 4.02131 8.42544 3.51211C8.32415 3.00291 8.37614 2.47511 8.57482 1.99546C8.7735 1.5158 9.10995 1.10583 9.54163 0.817392C9.97331 0.528953 10.4808 0.375 11 0.375C11.6962 0.375 12.3639 0.651562 12.8562 1.14384C13.3484 1.63613 13.625 2.30381 13.625 3ZM3.5 0.375C2.98083 0.375 2.47331 0.528953 2.04163 0.817392C1.60995 1.10583 1.2735 1.5158 1.07482 1.99546C0.876137 2.47511 0.824154 3.00291 0.92544 3.51211C1.02673 4.02131 1.27673 4.48904 1.64385 4.85616C2.01096 5.22327 2.47869 5.47328 2.98789 5.57456C3.49709 5.67585 4.02489 5.62386 4.50455 5.42518C4.9842 5.2265 5.39417 4.89005 5.68261 4.45837C5.97105 4.02669 6.125 3.51918 6.125 3C6.125 2.30381 5.84844 1.63613 5.35616 1.14384C4.86387 0.651562 4.19619 0.375 3.5 0.375ZM18.5 0.375C17.9808 0.375 17.4733 0.528953 17.0416 0.817392C16.61 1.10583 16.2735 1.5158 16.0748 1.99546C15.8761 2.47511 15.8242 3.00291 15.9254 3.51211C16.0267 4.02131 16.2767 4.48904 16.6438 4.85616C17.011 5.22327 17.4787 5.47328 17.9879 5.57456C18.4971 5.67585 19.0249 5.62386 19.5045 5.42518C19.9842 5.2265 20.3942 4.89005 20.6826 4.45837C20.971 4.02669 21.125 3.51918 21.125 3C21.125 2.65528 21.0571 2.31394 20.9252 1.99546C20.7933 1.67698 20.5999 1.3876 20.3562 1.14384C20.1124 0.900091 19.823 0.706735 19.5045 0.574816C19.1861 0.442897 18.8447 0.375 18.5 0.375Z"
            fill="#FF0828"
          />
        </svg>
      </button>
      <div class="dropdown">
        <div class="dropdown-heading"><span>Action</span></div>
        <ul>
          <li class="${partner.pinned === true ? "unpin" : "pin"} pin-button">
            <svg width="18" height="18" viewBox="0 0 21 21" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M20.3265 6.36269L14.6368 0.673936C14.4627 0.499605 14.2559 0.361307 14.0283 0.266949C13.8007 0.172591 13.5567 0.124023 13.3103 0.124023C13.0639 0.124023 12.8199 0.172591 12.5923 0.266949C12.3647 0.361307 12.1579 0.499605 11.9837 0.673936L7.11247 5.56206C5.94716 5.24987 3.67372 5.02581 1.32435 6.92144C1.11919 7.0865 0.951092 7.29292 0.831011 7.52727C0.71093 7.76161 0.641555 8.01862 0.627404 8.28156C0.613254 8.5445 0.654644 8.80747 0.748881 9.05334C0.843117 9.29922 0.988086 9.5225 1.17435 9.70862L5.43622 13.9714L1.70497 17.7027C1.49363 17.914 1.3749 18.2007 1.3749 18.4996C1.3749 18.7984 1.49363 19.0851 1.70497 19.2964C1.91632 19.5078 2.20296 19.6265 2.50185 19.6265C2.80073 19.6265 3.08738 19.5078 3.29872 19.2964L7.02997 15.5642L11.2928 19.8261C11.644 20.177 12.1201 20.3742 12.6165 20.3746C12.6597 20.3746 12.7037 20.3746 12.7478 20.3699C13.0154 20.3505 13.2757 20.274 13.5112 20.1454C13.7467 20.0168 13.9518 19.8392 14.1128 19.6246C14.5525 19.0405 15.144 18.1124 15.4665 16.9996C15.789 15.8867 15.79 14.8546 15.4815 13.8683L20.3284 9.0055C20.6771 8.65399 20.8726 8.17883 20.8722 7.68371C20.8719 7.18859 20.6757 6.7137 20.3265 6.36269ZM13.3647 12.8117C13.1982 12.9788 13.0885 13.194 13.051 13.4268C13.0135 13.6596 13.0501 13.8984 13.1556 14.1092C13.4772 14.7524 13.8025 16.078 12.5668 17.9202L3.06997 8.42425C5.04716 7.05644 6.77591 7.80175 6.91372 7.86175C7.12372 7.97078 7.36341 8.00879 7.59683 7.97008C7.83026 7.93137 8.04484 7.81803 8.20841 7.64706L13.3112 2.52737L18.4675 7.68362L13.3647 12.8117Z" fill="#2D2C2C" fill-opacity="0.2"/>
            </svg>
            <span>${partner.pinned === true ? "Unpin" : "Pin to the top"}</span>
          </li>
          <li class="${partner.blockStatus ? "block" : "unblock"}">
            <svg width="17" height="18" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M19.375 3.24971V8.76221C19.3859 10.2557 19.1395 11.7399 18.6466 13.1497C18.5968 13.2888 18.5202 13.4168 18.421 13.5263C18.3218 13.6358 18.202 13.7247 18.0685 13.7879C17.9349 13.8511 17.7902 13.8873 17.6427 13.8946C17.4951 13.9019 17.3476 13.8801 17.2084 13.8303C17.0693 13.7806 16.9414 13.7039 16.8319 13.6048C16.7224 13.5056 16.6335 13.3858 16.5703 13.2522C16.5071 13.1187 16.4708 12.974 16.4635 12.8264C16.4562 12.6789 16.4781 12.5313 16.5278 12.3922C16.9341 11.2255 17.1361 9.99754 17.125 8.76221V3.62471H8.24969C7.95132 3.62471 7.66517 3.50618 7.45419 3.29521C7.24321 3.08423 7.12469 2.79808 7.12469 2.49971C7.12469 2.20134 7.24321 1.91519 7.45419 1.70422C7.66517 1.49324 7.95132 1.37471 8.24969 1.37471H17.5C17.9973 1.37471 18.4742 1.57225 18.8258 1.92389C19.1775 2.27552 19.375 2.75243 19.375 3.24971ZM18.3325 17.4931C18.4319 17.6025 18.5088 17.7303 18.5588 17.8694C18.6088 18.0084 18.631 18.1559 18.624 18.3035C18.617 18.4511 18.5809 18.5959 18.518 18.7296C18.455 18.8633 18.3664 18.9833 18.257 19.0827C18.1477 19.1821 18.0199 19.259 17.8808 19.309C17.7418 19.359 17.5942 19.3812 17.4466 19.3741C17.299 19.3671 17.1543 19.3311 17.0206 19.2682C16.8869 19.2052 16.7669 19.1165 16.6675 19.0072L15.5669 17.796C14.1291 19.1229 12.4332 20.139 10.585 20.781C10.2051 20.9084 9.79397 20.9084 9.41406 20.781C7.94688 20.2916 0.625 17.421 0.625 8.76221V3.24971C0.625 2.95863 0.692772 2.67154 0.822949 2.41119C0.953126 2.15083 1.14213 1.92436 1.375 1.74971C1.37466 1.52261 1.44307 1.30073 1.57121 1.11324C1.69935 0.925755 1.88124 0.781437 2.09295 0.699269C2.30466 0.6171 2.53628 0.600928 2.75736 0.65288C2.97843 0.704831 3.17861 0.822473 3.33156 0.990336L18.3325 17.4931ZM14.0538 16.1319L2.875 3.83471V8.76221C2.875 15.6585 8.54969 18.0931 10 18.606C11.5038 18.0621 12.8825 17.2207 14.0538 16.1319Z" fill="#2D2C2C" fill-opacity="0.2"/>
            </svg>
            <span>${partner.blockStatus ? "Block" : "Unblock"}</span>
          </li>
          <li class="remove-button">
            <svg width="19" height="19" viewBox="0 0 22 22" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M15.875 11C15.875 11.2984 15.7565 11.5845 15.5455 11.7955C15.3345 12.0065 15.0484 12.125 14.75 12.125H7.25C6.95164 12.125 6.66549 12.0065 6.45451 11.7955C6.24353 11.5845 6.125 11.2984 6.125 11C6.125 10.7016 6.24353 10.4155 6.45451 10.2045C6.66549 9.99353 6.95164 9.875 7.25 9.875H14.75C15.0484 9.875 15.3345 9.99353 15.5455 10.2045C15.7565 10.4155 15.875 10.7016 15.875 11ZM21.125 11C21.125 13.0025 20.5312 14.9601 19.4186 16.6251C18.3061 18.2902 16.7248 19.5879 14.8747 20.3543C13.0246 21.1206 10.9888 21.3211 9.02471 20.9305C7.06066 20.5398 5.25656 19.5755 3.84055 18.1595C2.42454 16.7435 1.46023 14.9393 1.06955 12.9753C0.678878 11.0112 0.879387 8.97543 1.64572 7.12533C2.41206 5.27523 3.70981 3.69392 5.37486 2.58137C7.0399 1.46882 8.99747 0.875 11 0.875C13.6844 0.877978 16.258 1.94567 18.1562 3.84383C20.0543 5.74199 21.122 8.3156 21.125 11ZM18.875 11C18.875 9.44247 18.4131 7.91992 17.5478 6.62488C16.6825 5.32985 15.4526 4.32049 14.0136 3.72445C12.5747 3.12841 10.9913 2.97246 9.46367 3.27632C7.93607 3.58017 6.53288 4.3302 5.43154 5.43153C4.3302 6.53287 3.58018 7.93606 3.27632 9.46366C2.97246 10.9913 3.12841 12.5747 3.72445 14.0136C4.32049 15.4526 5.32985 16.6825 6.62489 17.5478C7.91993 18.4131 9.44248 18.875 11 18.875C13.0879 18.8728 15.0896 18.0424 16.566 16.566C18.0424 15.0896 18.8728 13.0879 18.875 11Z" fill="#FF0828"/>
            </svg>
            <span>Remove</span>
          </li>
        </ul>
      </div>
    </div>
  `;

  const createPinButton = () => `
    <button class="pin">
      <svg class="inactive" width="15" height="15" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M13.8844 4.90911L10.0913 1.11661C9.97516 1.00039 9.8373 0.908189 9.68555 0.845284C9.5338 0.782378 9.37115 0.75 9.20688 0.75C9.04261 0.75 8.87995 0.782378 8.7282 0.845284C8.57645 0.908189 8.43859 1.00039 8.3225 1.11661L5.075 4.37536C4.29813 4.16723 2.7825 4.01786 1.21625 5.28161C1.07948 5.39165 0.967415 5.52927 0.887361 5.6855C0.807307 5.84172 0.761057 6.01306 0.751623 6.18836C0.742189 6.36365 0.769783 6.53896 0.832608 6.70288C0.895432 6.8668 0.992078 7.01565 1.11625 7.13973L3.9575 9.98161L1.47 12.4691C1.32911 12.61 1.24995 12.8011 1.24995 13.0004C1.24995 13.1996 1.32911 13.3907 1.47 13.5316C1.6109 13.6725 1.80199 13.7517 2.00125 13.7517C2.20051 13.7517 2.39161 13.6725 2.5325 13.5316L5.02 11.0435L7.86188 13.8847C8.09604 14.1186 8.41341 14.2501 8.74438 14.2504C8.77313 14.2504 8.8025 14.2504 8.83188 14.2472C9.01028 14.2343 9.18383 14.1833 9.34082 14.0976C9.49781 14.0119 9.63457 13.8935 9.74188 13.7504C10.035 13.361 10.4294 12.7422 10.6444 12.0004C10.8594 11.2585 10.86 10.5704 10.6544 9.91286L13.8856 6.67098C14.1181 6.43664 14.2484 6.11987 14.2482 5.78979C14.2479 5.45971 14.1172 5.14312 13.8844 4.90911ZM9.24313 9.20848C9.13216 9.31986 9.05901 9.4633 9.034 9.61852C9.00899 9.77374 9.03339 9.93289 9.10375 10.0735C9.31813 10.5022 9.535 11.386 8.71125 12.6141L2.38 6.28348C3.69813 5.37161 4.85063 5.86848 4.9425 5.90848C5.0825 5.98117 5.24229 6.00651 5.39791 5.98071C5.55353 5.9549 5.69658 5.87934 5.80563 5.76536L9.2075 2.35223L12.645 5.78973L9.24313 9.20848Z" fill="#2D2C2C" fill-opacity="0.4"/>
      </svg>
      <svg class="active" width="13" height="13" viewBox="0 0 13 13" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M12.7082 5.49996L9.3663 8.85308C9.6513 9.64496 9.76942 10.9712 8.54192 12.6031C8.45607 12.7178 8.34646 12.8127 8.22058 12.8812C8.0947 12.9497 7.95552 12.9902 7.81255 13C7.7888 13 7.76567 13 7.74192 13C7.61052 13 7.4804 12.9741 7.359 12.9238C7.2376 12.8735 7.12731 12.7998 7.03442 12.7068L4.01817 9.68746L1.3538 12.3537C1.25998 12.4475 1.13273 12.5002 1.00005 12.5002C0.867364 12.5002 0.740116 12.4475 0.646295 12.3537C0.552475 12.2599 0.499767 12.1326 0.499767 12C0.499767 11.8673 0.552475 11.74 0.646295 11.6462L3.31255 8.98183L0.293796 5.96308C0.19457 5.86373 0.117379 5.7446 0.067248 5.61344C0.0171169 5.48228 -0.00483088 5.34203 0.00283378 5.20183C0.0104984 5.06162 0.0476035 4.92461 0.111734 4.79969C0.175864 4.67478 0.265581 4.56477 0.375045 4.47683C1.9638 3.19496 3.48442 3.44683 4.15005 3.65558L7.50005 0.293706C7.59291 0.200824 7.70316 0.127144 7.8245 0.0768747C7.94584 0.0266058 8.07589 0.000732422 8.20723 0.000732422C8.33857 0.000732422 8.46863 0.0266058 8.58997 0.0768747C8.71131 0.127144 8.82156 0.200824 8.91442 0.293706L12.7075 4.08621C12.895 4.27364 13.0004 4.52785 13.0005 4.79295C13.0006 5.05805 12.8955 5.31236 12.7082 5.49996Z" fill="#FF0828"/>
      </svg>
    </button>
  `;

  const attachRowEventListeners = (row, partner) => {
    const deliveryPartnerName = row.querySelector(
      "td:nth-child(3) > div > span:nth-child(1)"
    );
    if (partner.name) {
      deliveryPartnerName.addEventListener("click", () =>
        populateDeliveryPartnerModal(partner.id)
      );
      setupBlockButton(row, partner);
      setupPinButton(row, partner);
      setupRemoveButton(row, partner);
    }
  };

  const setupBlockButton = (row, partner) => {
    const blockModal = document.getElementById("block-modal");
    const blockButton =
      row.querySelector(".block") || row.querySelector(".unblock");
    let isBlocked = false;

    const setBlock = () => {
      isBlocked = restaurantData.blocked.blockedPartners.some(
        (item) => item.id === partner.id
      );
      blockModal.classList.toggle("block", !isBlocked);
      blockModal.classList.toggle("unblock", isBlocked);
      blockButton.classList.toggle("block", !isBlocked);
      blockButton.classList.toggle("unblock", isBlocked);
      blockButton.querySelector("span").textContent = isBlocked
        ? "Unblock"
        : "Block";
    };

    setBlock();

    blockButton.addEventListener("click", () => {
      toggleModalContent(isBlocked);
      blockModal.showModal();
    });

    const toggleModalContent = (isBlocked) => {
      blockModal.style.zIndex = 50;
      blockModal.innerHTML = isBlocked
        ? createUnblockModalContent()
        : createBlockModalContent();
      bindModalEventListeners(blockModal, partner);
    };

    const createUnblockModalContent = () => `
      <div class="modal-heading">
        <p><img src="./assets/icons/smiles.svg" alt="smile" /><span>Oh Yeah! 👍🏾</span></p>
        <button type="button" class="close-button"><img src="./assets/icons/close.svg" alt="close icon" class="close-icon" /></button>
      </div>
      <div class="modal-body">
        <h2>Great! <br/> Jackson will be able to run deliveries for your restaurant, when you unblock Delivery Partner.</h2>
        <p class="button-section">
          <button id="unblock-confirm-button" class="open-modal-btn"><span> Unblock Delivery Partner </span><img src="./assets/icons/double-check.svg" alt="check-icon"/></button>
          <button id="cancel-button">Cancel</button>
        </p>
      </div>
    `;

    const createBlockModalContent = () => `
      <div class="modal-heading">
        <p><img src="./assets/icons/frown.svg" alt="frown" /><span>Oh No!</span></p>
        <button type="button" class="close-button"><img src="./assets/icons/close.svg" alt="close icon" class="close-icon" /></button>
      </div>
      <div class="modal-body">
        <h2>Do you want to block this Delivery Partner?</h2>
        <p class="button-section">
          <button id="block-confirm-button" class="open-modal-btn"><span> Block Delivery Partner </span><img src="./assets/icons/block-icon.svg" alt="block-icon"/></button>
          <button id="cancel-button">Cancel</button>
        </p>
      </div>
    `;

    const bindModalEventListeners = (modal, partner) => {
      modal
        .querySelectorAll(".close-button, #cancel-button")
        .forEach((button) =>
          button.addEventListener("click", () => modal.close())
        );

      modal
        .querySelector("#block-confirm-button")
        ?.addEventListener("click", async () => {
          modal.close();
          await handleBlock(partner);
        });

      modal
        .querySelector("#unblock-confirm-button")
        ?.addEventListener("click", async () => {
          modal.close();
          await handleUnblock(partner);
        });
    };

    const handleBlock = async (partner) => {
      try {
        restaurantData.blocked.blockedPartners.push({
          id: partner.id,
          name: partner.name,
        });
        await update("restaurants", restaurantID, {
          blocked: restaurantData.blocked,
        });
        setBlock();
      } catch (error) {
        createAlert(
          "Couldn't block delivery partner",
          "./assets/icons/bin.svg",
          "red-5"
        );
      }
    };

    const handleUnblock = async (partner) => {
      try {
        restaurantData.blocked.blockedPartners =
          restaurantData.blocked.blockedPartners.filter(
            (p) => p.id !== partner.id
          );
        await update("restaurants", restaurantID, {
          blocked: restaurantData.blocked,
        });
        setBlock();
      } catch (error) {
        createAlert(
          "Couldn't unblock delivery partner",
          "./assets/icons/bin.svg",
          "red-5"
        );
      }
    };
  };

  const setupPinButton = (row, partner) => {
    const pinButton = row.querySelector(".pin-button");
    pinButton.addEventListener("click", () => togglePin(partner.id));
  };

  const togglePin = async (rowId) => {
    const rowData = deliveryPartnersData.find((row) => row.id === rowId);
    if (rowData) {
      rowData.pinned = !rowData.pinned;
      try {
        await update("delivery-partners", rowId, { pinned: rowData.pinned });
        populateDeliveryPartnerPage("in-house");
      } catch (error) {
        console.error("Failed to update pinned status:", error);
      }
    }
  };

  const setupRemoveButton = (row, partner) => {
    const removeButton = row.querySelector(".remove-button");
    removeButton.addEventListener("click", async () => {
      try {
        deliveryPartnersData = deliveryPartnersData.filter(
          (item) => item.id !== partner.id
        );
        data = data.filter((item) => item.id !== partner.id);
        await remove("delivery-partners", partner.id);
        populateDeliveryPartnerPage("in-house");
      } catch (error) {
        console.error("Failed to remove delivery partner:", error);
      }
    });
  };

  updateDataWithFilteredResults(data).then((updatedData) => {
    const sortedData = [...updatedData].sort((a, b) => b.pinned - a.pinned);
    renderTableRows(sortedData);
  });
};

const populateDeliveryPartnerPage = async (type) => {
  console.log(type);
  let response = await get("delivery-partners");
  deliveryPartnersData = response;
  populateFreelanceDeliveryTable(deliveryPartnersData);

  const partnerTable = document.querySelector("#freelance-table");
  const inhouseTable = document.querySelector("#inhouse-table");
  const addPartnerBtn = document.querySelector(
    "#delivery > header .add-partner-btn"
  );

  if (type === "freelance") {
    partnerTable.classList.add("show");
    inhouseTable.classList.remove("show");
    addPartnerBtn.classList.remove("show");
    deliveryPage = "freelance";
  } else {
    partnerTable.classList.remove("show");
    inhouseTable.classList.add("show");
    addPartnerBtn.classList.add("show");
    deliveryPage = "in-house";
  }
  inHouseDeliveryPartnersData = restaurantData.deliveryPartners;

  populateInhouseDeliveryTable(inHouseDeliveryPartnersData);
  populateDeliveryPartnerInfoCards();
};

const populateQRPage = async () => {
  const userName = document.querySelector(
    "#qr-code > div > div:nth-child(1) > div.section-one > p > span"
  );
  userName.textContent = restaurantData.ownersName;

  const businessUrl = document.querySelector(
    "#qr-code > div > div:nth-child(1) > div.section-one > div > p:nth-child(1)"
  );
  businessUrl.textContent = restaurantData.businessUrl;

  const businessLocation = document.querySelector(
    "#qr-code > div > div:nth-child(1) > div.section-one > div > p:nth-child(2) > span"
  );
  businessLocation.textContent = restaurantData.businessAddress;
  const businessPhone = document.querySelector(
    "#qr-code > div > div:nth-child(1) > div.section-one > div > p:nth-child(3) > span"
  );
  businessPhone.textContent = restaurantData.businessPhone;
};

const populateReviews = () => {
  const container = document.querySelector(
    "#review .review-content .review-container"
  );

  function populateContainer() {
    container
      .querySelectorAll(":not(template)")
      .forEach((child) => child.remove());

    if (!restaurantReviews) {
      const notice = document.createElement("p");
      notice.textContent = "No reviews have been made yet";
      container.appendChild(notice);
    } else {
      restaurantReviews.forEach(async (item) => {
        const customerData = await get("customers", item.customerID);
        let reviewItem = createReviewCard(item, customerData, true);
        container.appendChild(reviewItem);
      });
    }
  }
  populateContainer();
};

// Handle ratings page

// populate the ratings section
const populateRatings = (ratings) => {
  const totalRatingsElement = document.querySelector(
    "#review .rating-container .total-rating"
  );
  const averageRatingElement = document.querySelector(
    "#review .rating-container .average-rating"
  );
  const progressBars = document.querySelectorAll(
    "#review .rating-container .rating .rating-row .progress-bar"
  );
  const progressBarPercentage = document.querySelectorAll(
    "#review .rating-container .rating .rating-row .progress-percentage"
  );

  let totalCount = ratings.length;
  let totalSum = 0;
  const ratingCounts = [0, 0, 0, 0, 0];
  const roundRating = (rating) => {
    return rating < 0 ? 0 : Math.round(rating);
  };

  ratings.forEach(({ rating }) => {
    const roundedRating = roundRating(rating);
    totalSum += roundedRating;
    if (roundedRating >= 1 && roundedRating <= 5) {
      ratingCounts[roundedRating - 1]++;
    }
  });

  const averageRating = (totalSum / totalCount).toFixed(1);
  totalRatingsElement.textContent = totalCount;
  averageRatingElement.textContent = `${averageRating} / 5.0`;

  // Update progress bars
  ratingCounts.forEach((count, index) => {
    let progressBarIndex = 4 - index;
    console.log(progressBarIndex);
    const percentage = ((count / totalCount) * 100).toFixed(0);
    const progressBar = progressBars[progressBarIndex];
    progressBar.style.width = `${percentage}%`;
    progressBarPercentage[progressBarIndex].textContent = `${percentage}%`;
  });

  const stars = document.querySelectorAll(".header-info .star");
  const averageStars = Math.round(averageRating);
  stars.forEach((star, index) => {
    star.style.display = index < averageStars ? "inline" : "none";
  });

  animateProgressBars();
};

// animate progress bars
const animateProgressBars = () => {
  const progressBars = document.querySelectorAll(".progress-bar.animate");
  // Ensure a <style> tag exists in the document
  let styleSheet = document.querySelector("style");
  if (!styleSheet) {
    styleSheet = document.createElement("style");
    document.head.appendChild(styleSheet);
  }

  progressBars.forEach((bar, index) => {
    const targetWidth = bar.style.width;
    const animationName = `progressAnimation${index}`;

    const keyframes = `
          @keyframes ${animationName} {
              from { width: 0; }
              to { width: ${targetWidth}; }
          }
      `;
    styleSheet.sheet.insertRule(keyframes, styleSheet.sheet.cssRules.length);

    bar.style.animation = `${animationName} 1s ease forwards`;
    bar.offsetWidth;
  });
};

// populate reviews page
const populateReviewsPage = async () => {
  restaurantReviews = await getFiltered("reviews", {
    businessID: restaurantID,
  });

  const totalReviews =
    restaurantReviews.length < 9
      ? `0${restaurantReviews.length}`
      : restaurantReviews.length;

  const reviewsCounter = document.querySelector(
    "#review > header > p > span.count"
  );
  reviewsCounter.textContent = totalReviews;

  const navReviewsCounter = document.querySelector(
    "#dashboard-header .navigation-container2 > ul:nth-child(1) > li:nth-child(7) > a > span.count"
  );
  navReviewsCounter.textContent = totalReviews;

  populateRatings(restaurantReviews);
  populateReviews();
};

const populateDashboard = () => {
  populateHomePage();
  populateDishesPage();
  populateDeliveryPartnerPage();
  populateDiscountPages();
  populateQRPage();
  populateReviewsPage();
};

document.addEventListener("DOMContentLoaded", async () => {
  try {
    await loadData();
    const dashboardHeader = document.getElementById("dashboard-header");
    const loader = document.querySelector(".loader-container");

    restaurantData = await get("restaurants", restaurantID);

    localStorage.setItem("id", restaurantData.id);
    localStorage.setItem("ownersName", restaurantData.ownersName);
    localStorage.setItem("businessName", restaurantData.businessName);
    localStorage.setItem("status", restaurantData.businessStatus);

    populateDashboard();

    loader.classList.add("hide");
    dashboardHeader.classList.add("show");
    mainElement.classList.add("show");
  } catch (error) {
    document.getElementById("loading").textContent = "Error loading data";
  }
});

notification_button.addEventListener("click", () => {
  notification_button.parentElement.classList.toggle("open");
});

notification_closeButton.addEventListener("click", () => {
  notification_button.parentElement.classList.remove("open");
});

let navDropdownContainers = document.querySelectorAll(
  ".notification-panel .notification, .notification-panel .profile"
);

mainDoc.addEventListener("click", (e) => {
  const clickedElement = e.target;

  const isAnyDropdownOpen = Array.from(navDropdownContainers).some(
    (container) => container.classList.contains("open")
  );

  if (isAnyDropdownOpen) {
    navDropdownContainers.forEach((container) => {
      if (
        container.classList.contains("open") &&
        !container.contains(clickedElement)
      ) {
        container.classList.remove("open");
      }
    });
    return;
  }
});

profile_section.addEventListener("click", () => {
  profile_section.parentElement.classList.toggle("open");
});

// Manage select input boxes
let selectInputContainers = document.querySelectorAll(
  ".floating-controls.dropdown"
);

selectInputContainers.forEach((container) => {
  let dropdownButton = container.querySelector("button");
  dropdownButton.addEventListener("click", () => {
    container.classList.toggle("open");
  });

  let dropdownItems = container.querySelectorAll("ul li");
  dropdownItems.forEach((item) => {
    item.addEventListener("click", () => {
      container.classList.add("selected");
      dropdownButton.textContent = item.textContent;
      container.classList.remove("open");
    });
  });
});

mainDoc.addEventListener("click", (e) => {
  const clickedElement = e.target;

  const isAnyDropdownOpen = Array.from(selectInputContainers).some(
    (container) => container.classList.contains("open")
  );

  if (isAnyDropdownOpen) {
    selectInputContainers.forEach((container) => {
      if (
        container.classList.contains("open") &&
        !container.contains(clickedElement)
      ) {
        container.classList.remove("open");
      }
    });
    return;
  }
});

// Function to remove show class
function clearShowClass(elements) {
  elements.forEach((item) => item?.classList.remove("show"));
}

const sideBarBtn = document.querySelector(".side-nav-btn");
const sideBar = document.querySelector(".side-nav-container");
const sideBarCloseBtn = sideBar.querySelector(".close-btn");
sideBarBtn.addEventListener("click", () => {
  sideBar.classList.add("open");
});

sideBarCloseBtn.addEventListener("click", () => {
  sideBar.classList.remove("open");
});

// Handle page display
const handleShowMainContent = (id) => {
  const allContent = document.querySelectorAll(".main-content-div > div");
  const titleSpan = document.getElementById("page-title");

  allContent.forEach((content) => content.classList.remove("open"));

  const selectedElement = document.getElementById(id);
  const title = selectedElement
    ? selectedElement.getAttribute("data-title")
    : "Unknown";

  const titleMappings = {
    discount: {
      title: "Discount &nbsp; > &nbsp; Daily Discount",
      section: "discount",
      closeSection: "terms-of-use",
    },
    legal: {
      title: "Legal &nbsp; > &nbsp; Privacy Policy",
      openSection: "privacy-policy",
      closeSection: "terms-of-use",
    },
    "terms-of-use": {
      title: "Legal &nbsp; > &nbsp; Terms Of Use",
      openSection: "terms-of-use",
      closeSection: "privacy-policy",
    },
    delivery: {
      title: "Delivery Partners &nbsp; > &nbsp; In House",
      section: "delivery",
      populate: "in-house",
    },
    "in-house": {
      title: "Delivery Partners &nbsp; > &nbsp; In House",
      section: "delivery",
      populate: "in-house",
    },
    freelance: {
      title: "Delivery Partners &nbsp; > &nbsp; Freelance",
      section: "delivery",
      populate: "freelance",
    },
  };

  if (titleMappings[id]) {
    const mapping = titleMappings[id];
    titleSpan.innerHTML = typeof mapping === "string" ? mapping : mapping.title;

    if (mapping.section) {
      document.getElementById(mapping.section).classList.add("open");
      if (mapping.populate) {
        populateDeliveryPartnerPage(mapping.populate);
      }
    }

    if (mapping.openSection) {
      document.getElementById("legal").classList.add("open");
      document
        .querySelector(`#legal .${mapping.openSection}`)
        .classList.add("open");
      if (mapping.closeSection) {
        document
          .querySelector(`#legal .${mapping.closeSection}`)
          .classList.remove("open");
      }
    }
  } else {
    titleSpan.innerHTML = title;
    if (selectedElement) {
      selectedElement.classList.add("open");
    }
  }

  active_tab = id.toLowerCase();
};

const handleEnableMenu = () => {
  const allMenuItems = document.querySelectorAll(
    ".navigation-container2 > ul > li > a"
  );
  const deliveryNavigation = document.querySelector(
    ".navigation-container2 ul li.delivery-navigation"
  );

  allMenuItems.forEach((menuItem) => {
    menuItem.addEventListener("click", () => {
      const selectedId = menuItem.getAttribute("data-id");

      // Remove 'active' class from all menu items
      allMenuItems.forEach((item) => item.classList.remove("active"));

      // Define the mapping of IDs to menu indices
      const menuIndexMap = {
        legal: 11,
        "privacy-policy": 10,
        "terms-of-use": 10,
        delivery: 4,
        "in-house": 3,
        freelance: 3,
      };

      // Activate the appropriate menu item based on the selected ID
      const activeIndex = menuIndexMap[selectedId];
      if (activeIndex !== undefined) {
        allMenuItems[activeIndex].classList.add("active");
      }

      // Manage the delivery navigation visibility
      if (selectedId === "delivery") {
        deliveryNavigation.classList.add("open");
      } else {
        deliveryNavigation.classList.remove("open");
      }
      console.log(active_tab);
      // Add 'active' class to the clicked menu item and close the sidebar
      menuItem.classList.add("active");
      sideBar.classList.remove("open");

      handleShowMainContent(selectedId);
    });
  });
};

const handleMultipleRowCheck = () => {
  let input = document.querySelector(
    "#order-table-head > div:first-of-type > label > input"
  );
  input.addEventListener("change", () => {
    let allRowInput = document.querySelectorAll(
      "#order-table-body-container > div > div:first-of-type input"
    );
    let len = allRowInput.length;
    console.log(len, "len");
    for (let i = 0; i < len; i++) {
      const eachInput = allRowInput[i];
      const check = input.checked;
      if (check) {
        eachInput.checked = true;
      } else {
        eachInput.checked = false;
      }
    }
  });
};

const enableActionsDiv = () => {
  let allBtns = document.querySelectorAll(
    "#order-table-body-container > div > div:last-of-type > div > button"
  );
  let allDivs = document.querySelectorAll(
    "#order-table-body-container > div > div:last-of-type > div"
  );
  let len = allDivs.length;
  for (let i = 0; i < len; i++) {
    let btn = allBtns[i];
    btn.onclick = () => {
      let parent = btn.parentElement;
      allDivs.forEach((e) => {
        e.classList.remove("active");
      });
      parent.classList.add("active");
    };
  }
};

const removeModalsOnClick = () => {
  window.addEventListener("click", (e) => {
    let allDivs = document.querySelectorAll(
      "#order-table-body-container > div > div:last-of-type > div"
    );
    let ele = e.target;
    let check = false;
    let last = false;
    for (let i = 0; i < 3; i++) {
      check =
        ele.classList.contains("actions-div") ||
        ele.getAttribute("data-id") === "svg";
      if (check) {
        break;
      }
      if (i == 2) {
        last = true;
      }
      ele = ele?.parentElement;
    }
    if (last) {
      allDivs.forEach((e) => {
        e.classList.remove("active");
      });
    }
  });
};

const highlightNewOrder = () => {
  let allRows = document.querySelectorAll("#order-table-body-container > div");
  let allNewRows = [];
  for (let i = 0; i < allRows.length; i++) {
    if (allRows[i].classList.contains("new")) {
      allNewRows.push(allRows[i]);
    }
  }
  let len = allNewRows.length;
  setInterval(() => {
    for (let i = 0; i < len; i++) {
      allNewRows[i].style.backgroundColor = "rgba(255, 8, 40, 0.05)";
    }
    setTimeout(() => {
      for (let i = 0; i < len; i++) {
        allNewRows[i].style.backgroundColor = "rgba(45, 44, 44, 0.02)";
      }
    }, 3000);
  }, 4000);
};

const generateDummyOrders = () => {
  return {
    id: `ZP${Math.floor(Math.random() * 900000000000000) + 1}`,
    order_type: Math.floor(Math.random() * 3) + 1,
    // 1 - delivery
    // 2 - takeaway
    // 3 - dine in
    name: order_sample_names[`${Math.floor(Math.random() * 12)}`],
    mobile: `+1${Math.floor(Math.random() * 900000000) + 1}`,
    location: order_sample_address[Math.floor(Math.random() * 5) + 1],
    time: Math.floor(Math.random() * 23) + 1,
    status: Math.floor(Math.random() * 5) + 1,
    // Delivery
    // 1 - new
    // 2 - preparing  // complete order
    // 3 - in-transit
    // 4 - completed
    // 6 - decline
    avatar: Math.floor(Math.random() * 3) + 1,
  };
};

const createDummyOrders = () => {
  let end = 20;
  all_orders = [];
  for (let i = 0; i < end; i++) {
    all_orders.push(generateDummyOrders());
  }
  let newOrders = all_orders.filter((e) => e.status === 1);
  let otherOrders = all_orders.filter((e) => e.status !== 1);
  all_orders = [...newOrders, ...otherOrders];
};

const loadOrders = (t_orders = null, replace = true) => {
  let temp_orders = all_orders;
  let orders = [];

  if (t_orders && replace) {
    temp_orders = t_orders;
    all_orders = temp_orders;
    orders = all_orders;
  } else if (t_orders && !replace) {
    temp_orders = t_orders;
    orders = temp_orders;
  } else {
    orders = temp_orders;
  }
  let len = orders.length;
  let orderTableDiv = document.getElementById("order-table");
  let noOrderContainer = document.getElementById("no-order-container");
  if (len > 0) {
    let container = document.getElementById("order-table-body-container");
    let content = "";
    for (let i = 0; i < len; i++) {
      content += `
        <div class="order-table-row ${orders[i].status == 1 ? "new" : ""}">
          <div>
            <!--
              <label>
                <input type="checkbox">
                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M10 0.25C8.07164 0.25 6.18657 0.821828 4.58319 1.89317C2.97982 2.96451 1.73013 4.48726 0.992179 6.26884C0.254225 8.05042 0.061142 10.0108 0.437348 11.9021C0.813554 13.7934 1.74215 15.5307 3.10571 16.8943C4.46928 18.2579 6.20656 19.1865 8.09787 19.5627C9.98919 19.9389 11.9496 19.7458 13.7312 19.0078C15.5127 18.2699 17.0355 17.0202 18.1068 15.4168C19.1782 13.8134 19.75 11.9284 19.75 10C19.7473 7.41498 18.7192 4.93661 16.8913 3.10872C15.0634 1.28084 12.585 0.25273 10 0.25ZM14.2806 8.28063L9.03063 13.5306C8.96097 13.6004 8.87826 13.6557 8.78721 13.6934C8.69616 13.7312 8.59857 13.7506 8.5 13.7506C8.40144 13.7506 8.30385 13.7312 8.2128 13.6934C8.12175 13.6557 8.03903 13.6004 7.96938 13.5306L5.71938 11.2806C5.57865 11.1399 5.49959 10.949 5.49959 10.75C5.49959 10.551 5.57865 10.3601 5.71938 10.2194C5.86011 10.0786 6.05098 9.99958 6.25 9.99958C6.44903 9.99958 6.6399 10.0786 6.78063 10.2194L8.5 11.9397L13.2194 7.21937C13.2891 7.14969 13.3718 7.09442 13.4628 7.0567C13.5539 7.01899 13.6515 6.99958 13.75 6.99958C13.8486 6.99958 13.9461 7.01899 14.0372 7.0567C14.1282 7.09442 14.2109 7.14969 14.2806 7.21937C14.3503 7.28906 14.4056 7.37178 14.4433 7.46283C14.481 7.55387 14.5004 7.65145 14.5004 7.75C14.5004 7.84855 14.481 7.94613 14.4433 8.03717C14.4056 8.12822 14.3503 8.21094 14.2806 8.28063Z" fill="none" fill-opacity="0.6"/>
                </svg>
              </label>
            -->
          </div>
          <div>
            ${order_icons[orders[i].order_type - 1]}
            <span>${order_sample_types[orders[i].order_type - 1]}</span>
          </div>
          <div><span class="open-modal-btn" onclick="handleDisplayOrderDetail('${
            orders[i].id
          }')">${orders[i].id}</span></div>
          <div>
            <img src="${order_sample_avatars[orders[i].avatar]}" alt="Avatar" />
            <span>${orders[i].name}</span>
          </div>
          <div><a href="tel:${orders[i].mobile}">${orders[i].mobile}</a></div>
          <div><a target="_blank" href="http://maps.google.com/?q=${
            orders[i].location
          }">${orders[i].location}</a></div>
          <div>${orders[i].time} hour ago</div>
          <div>
            ${
              orders[i].status === 1
                ? `<button class="cancel" onclick="handleOrderProcess('${orders[i].id}',5)">Decline</button> <button class="accept" onclick="handleOrderProcess('${orders[i].id}',2)">Accept</button>`
                : orders[i].status === 2 && orders[i].order_type === 1
                ? `
                <button class="preparing" onclick="handleOrderProcess('${orders[i].id}',3)">Preparing</button>
                <button class="print">
                  <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M15.094 4.78125H14.3438V2.8125C14.3438 2.58872 14.2549 2.37411 14.0966 2.21588C13.9384 2.05764 13.7238 1.96875 13.5 1.96875H4.5C4.27622 1.96875 4.06161 2.05764 3.90338 2.21588C3.74514 2.37411 3.65625 2.58872 3.65625 2.8125V4.78125H2.90602C1.76906 4.78125 0.84375 5.66438 0.84375 6.75V12.375C0.84375 12.5988 0.932645 12.8134 1.09088 12.9716C1.24911 13.1299 1.46372 13.2188 1.6875 13.2188H3.65625V15.1875C3.65625 15.4113 3.74514 15.6259 3.90338 15.7841C4.06161 15.9424 4.27622 16.0312 4.5 16.0312H13.5C13.7238 16.0312 13.9384 15.9424 14.0966 15.7841C14.2549 15.6259 14.3438 15.4113 14.3438 15.1875V13.2188H16.3125C16.5363 13.2188 16.7509 13.1299 16.9091 12.9716C17.0674 12.8134 17.1562 12.5988 17.1562 12.375V6.75C17.1562 5.66438 16.2309 4.78125 15.094 4.78125ZM5.34375 3.65625H12.6562V4.78125H5.34375V3.65625ZM12.6562 14.3438H5.34375V12.0938H12.6562V14.3438ZM15.4688 11.5312H14.3438V11.25C14.3438 11.0262 14.2549 10.8116 14.0966 10.6534C13.9384 10.4951 13.7238 10.4062 13.5 10.4062H4.5C4.27622 10.4062 4.06161 10.4951 3.90338 10.6534C3.74514 10.8116 3.65625 11.0262 3.65625 11.25V11.5312H2.53125V6.75C2.53125 6.59742 2.70281 6.46875 2.90602 6.46875H15.094C15.2972 6.46875 15.4688 6.59742 15.4688 6.75V11.5312ZM14.3438 8.4375C14.3438 8.66 14.2778 8.87751 14.1542 9.06252C14.0305 9.24752 13.8548 9.39172 13.6493 9.47686C13.4437 9.56201 13.2175 9.58429 12.9993 9.54088C12.781 9.49748 12.5806 9.39033 12.4233 9.233C12.2659 9.07566 12.1588 8.87521 12.1154 8.65698C12.072 8.43875 12.0942 8.21255 12.1794 8.00698C12.2645 7.80141 12.4087 7.62571 12.5937 7.5021C12.7787 7.37848 12.9962 7.3125 13.2188 7.3125C13.5171 7.3125 13.8033 7.43103 14.0142 7.642C14.2252 7.85298 14.3438 8.13913 14.3438 8.4375Z" fill="#0075FF"/>
                  </svg>
                </button>
              `
                : orders[i].status === 2 && orders[i].order_type !== 1
                ? `
              <button class="to-complete" onclick="handleOrderProcess('${orders[i].id}',4)">Complete Order</button>
                <button class="print">
                  <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M15.094 4.78125H14.3438V2.8125C14.3438 2.58872 14.2549 2.37411 14.0966 2.21588C13.9384 2.05764 13.7238 1.96875 13.5 1.96875H4.5C4.27622 1.96875 4.06161 2.05764 3.90338 2.21588C3.74514 2.37411 3.65625 2.58872 3.65625 2.8125V4.78125H2.90602C1.76906 4.78125 0.84375 5.66438 0.84375 6.75V12.375C0.84375 12.5988 0.932645 12.8134 1.09088 12.9716C1.24911 13.1299 1.46372 13.2188 1.6875 13.2188H3.65625V15.1875C3.65625 15.4113 3.74514 15.6259 3.90338 15.7841C4.06161 15.9424 4.27622 16.0312 4.5 16.0312H13.5C13.7238 16.0312 13.9384 15.9424 14.0966 15.7841C14.2549 15.6259 14.3438 15.4113 14.3438 15.1875V13.2188H16.3125C16.5363 13.2188 16.7509 13.1299 16.9091 12.9716C17.0674 12.8134 17.1562 12.5988 17.1562 12.375V6.75C17.1562 5.66438 16.2309 4.78125 15.094 4.78125ZM5.34375 3.65625H12.6562V4.78125H5.34375V3.65625ZM12.6562 14.3438H5.34375V12.0938H12.6562V14.3438ZM15.4688 11.5312H14.3438V11.25C14.3438 11.0262 14.2549 10.8116 14.0966 10.6534C13.9384 10.4951 13.7238 10.4062 13.5 10.4062H4.5C4.27622 10.4062 4.06161 10.4951 3.90338 10.6534C3.74514 10.8116 3.65625 11.0262 3.65625 11.25V11.5312H2.53125V6.75C2.53125 6.59742 2.70281 6.46875 2.90602 6.46875H15.094C15.2972 6.46875 15.4688 6.59742 15.4688 6.75V11.5312ZM14.3438 8.4375C14.3438 8.66 14.2778 8.87751 14.1542 9.06252C14.0305 9.24752 13.8548 9.39172 13.6493 9.47686C13.4437 9.56201 13.2175 9.58429 12.9993 9.54088C12.781 9.49748 12.5806 9.39033 12.4233 9.233C12.2659 9.07566 12.1588 8.87521 12.1154 8.65698C12.072 8.43875 12.0942 8.21255 12.1794 8.00698C12.2645 7.80141 12.4087 7.62571 12.5937 7.5021C12.7787 7.37848 12.9962 7.3125 13.2188 7.3125C13.5171 7.3125 13.8033 7.43103 14.0142 7.642C14.2252 7.85298 14.3438 8.13913 14.3438 8.4375Z" fill="#0075FF"/>
                  </svg>
                </button>
              
              `
                : orders[i].status === 3
                ? `
                <button class="in-transit">In Transit</button>
                <button class="print">
                  <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M15.094 4.78125H14.3438V2.8125C14.3438 2.58872 14.2549 2.37411 14.0966 2.21588C13.9384 2.05764 13.7238 1.96875 13.5 1.96875H4.5C4.27622 1.96875 4.06161 2.05764 3.90338 2.21588C3.74514 2.37411 3.65625 2.58872 3.65625 2.8125V4.78125H2.90602C1.76906 4.78125 0.84375 5.66438 0.84375 6.75V12.375C0.84375 12.5988 0.932645 12.8134 1.09088 12.9716C1.24911 13.1299 1.46372 13.2188 1.6875 13.2188H3.65625V15.1875C3.65625 15.4113 3.74514 15.6259 3.90338 15.7841C4.06161 15.9424 4.27622 16.0312 4.5 16.0312H13.5C13.7238 16.0312 13.9384 15.9424 14.0966 15.7841C14.2549 15.6259 14.3438 15.4113 14.3438 15.1875V13.2188H16.3125C16.5363 13.2188 16.7509 13.1299 16.9091 12.9716C17.0674 12.8134 17.1562 12.5988 17.1562 12.375V6.75C17.1562 5.66438 16.2309 4.78125 15.094 4.78125ZM5.34375 3.65625H12.6562V4.78125H5.34375V3.65625ZM12.6562 14.3438H5.34375V12.0938H12.6562V14.3438ZM15.4688 11.5312H14.3438V11.25C14.3438 11.0262 14.2549 10.8116 14.0966 10.6534C13.9384 10.4951 13.7238 10.4062 13.5 10.4062H4.5C4.27622 10.4062 4.06161 10.4951 3.90338 10.6534C3.74514 10.8116 3.65625 11.0262 3.65625 11.25V11.5312H2.53125V6.75C2.53125 6.59742 2.70281 6.46875 2.90602 6.46875H15.094C15.2972 6.46875 15.4688 6.59742 15.4688 6.75V11.5312ZM14.3438 8.4375C14.3438 8.66 14.2778 8.87751 14.1542 9.06252C14.0305 9.24752 13.8548 9.39172 13.6493 9.47686C13.4437 9.56201 13.2175 9.58429 12.9993 9.54088C12.781 9.49748 12.5806 9.39033 12.4233 9.233C12.2659 9.07566 12.1588 8.87521 12.1154 8.65698C12.072 8.43875 12.0942 8.21255 12.1794 8.00698C12.2645 7.80141 12.4087 7.62571 12.5937 7.5021C12.7787 7.37848 12.9962 7.3125 13.2188 7.3125C13.5171 7.3125 13.8033 7.43103 14.0142 7.642C14.2252 7.85298 14.3438 8.13913 14.3438 8.4375Z" fill="#0075FF"/>
                  </svg>
                </button>
              `
                : orders[i].status === 4
                ? `
                <button class="complete">Completed</button>
                <button class="print">
                  <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M15.094 4.78125H14.3438V2.8125C14.3438 2.58872 14.2549 2.37411 14.0966 2.21588C13.9384 2.05764 13.7238 1.96875 13.5 1.96875H4.5C4.27622 1.96875 4.06161 2.05764 3.90338 2.21588C3.74514 2.37411 3.65625 2.58872 3.65625 2.8125V4.78125H2.90602C1.76906 4.78125 0.84375 5.66438 0.84375 6.75V12.375C0.84375 12.5988 0.932645 12.8134 1.09088 12.9716C1.24911 13.1299 1.46372 13.2188 1.6875 13.2188H3.65625V15.1875C3.65625 15.4113 3.74514 15.6259 3.90338 15.7841C4.06161 15.9424 4.27622 16.0312 4.5 16.0312H13.5C13.7238 16.0312 13.9384 15.9424 14.0966 15.7841C14.2549 15.6259 14.3438 15.4113 14.3438 15.1875V13.2188H16.3125C16.5363 13.2188 16.7509 13.1299 16.9091 12.9716C17.0674 12.8134 17.1562 12.5988 17.1562 12.375V6.75C17.1562 5.66438 16.2309 4.78125 15.094 4.78125ZM5.34375 3.65625H12.6562V4.78125H5.34375V3.65625ZM12.6562 14.3438H5.34375V12.0938H12.6562V14.3438ZM15.4688 11.5312H14.3438V11.25C14.3438 11.0262 14.2549 10.8116 14.0966 10.6534C13.9384 10.4951 13.7238 10.4062 13.5 10.4062H4.5C4.27622 10.4062 4.06161 10.4951 3.90338 10.6534C3.74514 10.8116 3.65625 11.0262 3.65625 11.25V11.5312H2.53125V6.75C2.53125 6.59742 2.70281 6.46875 2.90602 6.46875H15.094C15.2972 6.46875 15.4688 6.59742 15.4688 6.75V11.5312ZM14.3438 8.4375C14.3438 8.66 14.2778 8.87751 14.1542 9.06252C14.0305 9.24752 13.8548 9.39172 13.6493 9.47686C13.4437 9.56201 13.2175 9.58429 12.9993 9.54088C12.781 9.49748 12.5806 9.39033 12.4233 9.233C12.2659 9.07566 12.1588 8.87521 12.1154 8.65698C12.072 8.43875 12.0942 8.21255 12.1794 8.00698C12.2645 7.80141 12.4087 7.62571 12.5937 7.5021C12.7787 7.37848 12.9962 7.3125 13.2188 7.3125C13.5171 7.3125 13.8033 7.43103 14.0142 7.642C14.2252 7.85298 14.3438 8.13913 14.3438 8.4375Z" fill="#0075FF"/>
                  </svg>
                </button>
              `
                : `
                <button class="decline">Declined</button>
                <button class="print">
                  <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M15.094 4.78125H14.3438V2.8125C14.3438 2.58872 14.2549 2.37411 14.0966 2.21588C13.9384 2.05764 13.7238 1.96875 13.5 1.96875H4.5C4.27622 1.96875 4.06161 2.05764 3.90338 2.21588C3.74514 2.37411 3.65625 2.58872 3.65625 2.8125V4.78125H2.90602C1.76906 4.78125 0.84375 5.66438 0.84375 6.75V12.375C0.84375 12.5988 0.932645 12.8134 1.09088 12.9716C1.24911 13.1299 1.46372 13.2188 1.6875 13.2188H3.65625V15.1875C3.65625 15.4113 3.74514 15.6259 3.90338 15.7841C4.06161 15.9424 4.27622 16.0312 4.5 16.0312H13.5C13.7238 16.0312 13.9384 15.9424 14.0966 15.7841C14.2549 15.6259 14.3438 15.4113 14.3438 15.1875V13.2188H16.3125C16.5363 13.2188 16.7509 13.1299 16.9091 12.9716C17.0674 12.8134 17.1562 12.5988 17.1562 12.375V6.75C17.1562 5.66438 16.2309 4.78125 15.094 4.78125ZM5.34375 3.65625H12.6562V4.78125H5.34375V3.65625ZM12.6562 14.3438H5.34375V12.0938H12.6562V14.3438ZM15.4688 11.5312H14.3438V11.25C14.3438 11.0262 14.2549 10.8116 14.0966 10.6534C13.9384 10.4951 13.7238 10.4062 13.5 10.4062H4.5C4.27622 10.4062 4.06161 10.4951 3.90338 10.6534C3.74514 10.8116 3.65625 11.0262 3.65625 11.25V11.5312H2.53125V6.75C2.53125 6.59742 2.70281 6.46875 2.90602 6.46875H15.094C15.2972 6.46875 15.4688 6.59742 15.4688 6.75V11.5312ZM14.3438 8.4375C14.3438 8.66 14.2778 8.87751 14.1542 9.06252C14.0305 9.24752 13.8548 9.39172 13.6493 9.47686C13.4437 9.56201 13.2175 9.58429 12.9993 9.54088C12.781 9.49748 12.5806 9.39033 12.4233 9.233C12.2659 9.07566 12.1588 8.87521 12.1154 8.65698C12.072 8.43875 12.0942 8.21255 12.1794 8.00698C12.2645 7.80141 12.4087 7.62571 12.5937 7.5021C12.7787 7.37848 12.9962 7.3125 13.2188 7.3125C13.5171 7.3125 13.8033 7.43103 14.0142 7.642C14.2252 7.85298 14.3438 8.13913 14.3438 8.4375Z" fill="#0075FF"/>
                  </svg>
                </button>
              `
            }   
          </div>
          <div>
            <div class="actions-div">
              <button>
                <svg data-id="svg" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M14.625 12C14.625 12.5192 14.471 13.0267 14.1826 13.4584C13.8942 13.8901 13.4842 14.2265 13.0045 14.4252C12.5249 14.6239 11.9971 14.6758 11.4879 14.5746C10.9787 14.4733 10.511 14.2233 10.1438 13.8562C9.77673 13.489 9.52673 13.0213 9.42544 12.5121C9.32415 12.0029 9.37614 11.4751 9.57482 10.9955C9.7735 10.5158 10.11 10.1058 10.5416 9.81739C10.9733 9.52895 11.4808 9.375 12 9.375C12.6962 9.375 13.3639 9.65156 13.8562 10.1438C14.3484 10.6361 14.625 11.3038 14.625 12ZM4.5 9.375C3.98083 9.375 3.47331 9.52895 3.04163 9.81739C2.60995 10.1058 2.2735 10.5158 2.07482 10.9955C1.87614 11.4751 1.82415 12.0029 1.92544 12.5121C2.02673 13.0213 2.27673 13.489 2.64385 13.8562C3.01096 14.2233 3.47869 14.4733 3.98789 14.5746C4.49709 14.6758 5.02489 14.6239 5.50455 14.4252C5.9842 14.2265 6.39417 13.8901 6.68261 13.4584C6.97105 13.0267 7.125 12.5192 7.125 12C7.125 11.3038 6.84844 10.6361 6.35616 10.1438C5.86387 9.65156 5.19619 9.375 4.5 9.375ZM19.5 9.375C18.9808 9.375 18.4733 9.52895 18.0416 9.81739C17.61 10.1058 17.2735 10.5158 17.0748 10.9955C16.8761 11.4751 16.8242 12.0029 16.9254 12.5121C17.0267 13.0213 17.2767 13.489 17.6438 13.8562C18.011 14.2233 18.4787 14.4733 18.9879 14.5746C19.4971 14.6758 20.0249 14.6239 20.5045 14.4252C20.9842 14.2265 21.3942 13.8901 21.6826 13.4584C21.971 13.0267 22.125 12.5192 22.125 12C22.125 11.6553 22.0571 11.3139 21.9252 10.9955C21.7933 10.677 21.5999 10.3876 21.3562 10.1438C21.1124 9.90009 20.823 9.70673 20.5045 9.57482C20.1861 9.4429 19.8447 9.375 19.5 9.375Z" fill="#FF0828"/>
                </svg>
              </button>
              <ul>
                <li>Actions</li>
                <li>View</li>
                <li>Complete Order</li>
                <li>Decline & Refund</li>
                <li>Print</li>
              </ul>
            </div>
          </div>
        </div>
      `;
    }
    container.innerHTML = content;
    // handleMultipleRowCheck();
    enableActionsDiv();
    removeModalsOnClick();
    highlightNewOrder();
  } else {
    orderTableDiv.style.display = "none";
    noOrderContainer.style.display = "flex";
  }
};

const handleOrderProcess = (id, n_order_status = null) => {
  let new_order_status = n_order_status;
  let type = 2;
  let temp_orders = all_orders.map((e) => {
    if (id === e.id) {
      type = e.order_type;
      let order_status = e.status;
      new_order_status = n_order_status
        ? n_order_status
        : order_status >= 4
        ? order_status
        : order_status + 1;
      if (type === 1 && new_order_status > 2) return e;
      return { ...e, status: new_order_status };
    }
    return e;
  });
  if (type === 1 && new_order_status > 2) return;
  loadOrders(temp_orders);
};

const handleClickSearchBar = () => {
  //function for when the searchbar is clicked
  let searchBars = document.querySelectorAll(
    "#main-search, #dashboard-header .notification-panel .search-button"
  );
  searchBars.forEach((item) => {
    item.addEventListener("click", () => {
      if (
        active_tab.toLocaleLowerCase() === "orders-div".toLowerCase() ||
        active_tab.toLocaleLowerCase() === "delivery".toLowerCase() ||
        active_tab.toLocaleLowerCase() === "in-house".toLowerCase() ||
        active_tab.toLocaleLowerCase() === "freelance".toLowerCase()
      ) {
        //doing nothing
      } else {
        let searchModal = document.getElementById("search-modal");
        searchModal.showModal();

        let searchModalCloseBtn = searchModal.querySelector(
          ".modal-heading .close-button"
        );
        searchModalCloseBtn.addEventListener("click", () => {
          searchModal.close();
        });
      }
    });
  });
};

const handleSearchInput = () => {
  // function for when user type on search input
  let search_input = document.querySelector("#main-search > input");
  search_input.addEventListener("input", () => {
    if (active_tab.toLocaleLowerCase() === "orders-div".toLowerCase()) {
      handleSearchOrder(search_input); // search for orders tab
    }
  });
};

const handleSearchOrder = (ele) => {
  // search function for orders tab
  let value = ele?.value;
  let temp_orders = all_orders;
  if (value) {
    let new_orders = temp_orders.filter((e) => {
      return `${e.id}${e.name}${e.mobile}${e.location}`
        .toLowerCase()
        .includes(value.toLowerCase());
    });
    console.log(new_orders);
    loadOrders(new_orders, false);
  } else {
    loadOrders(temp_orders);
  }
};

const handleDisplayOrderDetail = (order_id) => {
  console.log(order_id);
  let allModal = document.querySelectorAll(".modal");
  for (let i = 0; i < allModal.length; i++) {
    // allModal[i].classList.remove('open');
  }
  populateOrderModal(order_id || "ZP37646739299399");
};

window.addEventListener("load", () => {
  handleEnableMenu();
  handleShowMainContent("home");
  createDummyOrders();
  // loadOrders();
  enableActionsDiv();
  removeModalsOnClick();
  highlightNewOrder();
  handleClickSearchBar();
  handleSearchInput();
});

// Toggle switch
const toggle_switches = document.querySelectorAll(".toggle-button");

toggle_switches.forEach((item) => {
  item.addEventListener("click", () => {
    const button_text = item.querySelector(".button-text");
    const button_switch = item.querySelector("img");

    if (item.classList.contains("enabled")) {
      item.classList.replace("enabled", "disabled");
      button_text.textContent = "Disabled";
      button_switch.src = "./assets/icons/toggle-off.svg";
    } else if (item.classList.contains("disabled")) {
      item.classList.replace("disabled", "enabled");
      const button_text = item.querySelector(".button-text");
      button_text.textContent = "Enabled";

      button_switch.src = "./assets/icons/toggle-on.svg";
    }
  });
});

const toggle_containers = document.querySelectorAll(".toggle-container");
const toggle_container_buttons = document.querySelectorAll(
  ".toggle-container .head button"
);

toggle_container_buttons.forEach((item, index) => {
  item.addEventListener("click", () => {
    const container = toggle_containers[index];
    const button_switch = container.querySelector(".head button img");

    if (container.classList.contains("enabled")) {
      container.classList.replace("enabled", "disabled");
      button_switch.src = "./assets/icons/toggle-off.svg";
    } else if (container.classList.contains("disabled")) {
      container.classList.replace("disabled", "enabled");
      button_switch.src = "./assets/icons/toggle-on.svg";
    }
  });
});

// Select all progress bars
// function updateRatings(ratingsData) {
//   // Update total ratings
//   document.querySelector(".total-rating").textContent = ratingsData.totalRatings;

//   // Update average rating
//   document.querySelector(".header-info span:last-child").textContent = `${ratingsData.averageRating} / 5.0`;

//   // Get total count of all ratings
//   const totalVotes = Object.values(ratingsData.ratingCounts).reduce((a, b) => a + b, 0);

//   // Update progress bars
//   document.querySelectorAll(".rating-row").forEach((row) => {
//     const starValue = row.querySelector("p span").textContent.trim(); // Get star value (1-5)
//     const percentage = (ratingsData.ratingCounts[starValue] / totalVotes) * 100 || 0;

//     const progressBar = row.querySelector(".progress-bar");
//     progressBar.style.width = `${percentage}%`;
//     progressBar.textContent = `${percentage.toFixed(1)}%`;

//     // Animate Progress Bar
//     progressBar.style.animation = "none";
//     void progressBar.offsetWidth; // Force reflow
//     progressBar.style.animation = "progressAnim 1s ease forwards";
//   });
// }

// // Run function with provided ratings data
// updateRatings(ratingsData);

// Set Cropper js class
class ImageEdit {
  constructor({ inputFieldId, editModalId, onEdit, onFileChange, cropperNo }) {
    // Initialize instance properties
    this.inputFieldId = inputFieldId;
    this.editModalId = editModalId;
    this.onEdit = onEdit;
    this.onFileChange = onFileChange;
    this.cropper = null;
    this.file = null;
    this.isProcessing = false;
    this.processingQueue = Promise.resolve();

    // Get DOM elements
    this.inputField = document.getElementById(inputFieldId);
    this.editModal = document.getElementById(editModalId);

    if (!this.inputField || !this.editModal) {
      console.error(
        "Required elements not found. Check inputFieldId and editModalId."
      );
      return;
    }

    this.editImageWorkspace = this.editModal.querySelector(
      ".image-edit-workspace img"
    );
    if (!this.editImageWorkspace) {
      console.error("Image workspace element not found.");
      return;
    }

    // Get control elements
    this.rotateX180Button = this.editModal.querySelector(
      ".horizontal-flip-button"
    );
    this.rotateY180Button = this.editModal.querySelector(
      ".vertical-flip-button"
    );
    this.editCancelButton = this.editModal.querySelector(".cancel-button");
    this.editResetButton = this.editModal.querySelector(".reset-button");
    this.editDoneButton = this.editModal.querySelector(".save-button");
    this.rotateSlider = this.editModal.querySelector(".rotate-slider");
    this.fitHorizontalButton =
      this.editModal.querySelector(".horizontal-resize");
    this.fitVerticalButton = this.editModal.querySelector(".vertical-resize");
    this.degreeElement = this.editModal.querySelector(
      ".degree-container .degree"
    );
    this.rotate90Button = this.editModal.querySelector(".rotate-90-button");

    // Initialize event listeners
    this.setupEventListeners();
  }

  setupEventListeners() {
    // Input field listener
    this.inputField.addEventListener("change", (e) => {
      this.processImage(e);
    });

    // Button listeners
    this.rotateX180Button?.addEventListener("click", () => {
      if (!this.cropper) return;
      const { scaleX } = this.cropper.getData();
      this.cropper.scaleX(scaleX === -1 ? 1 : -1);
    });

    this.rotateY180Button?.addEventListener("click", () => {
      if (!this.cropper) return;
      const { scaleY } = this.cropper.getData();
      this.cropper.scaleY(scaleY === -1 ? 1 : -1);
    });

    this.editCancelButton?.addEventListener("click", () => {
      // Don't destroy the cropper on cancel, just hide the modal
      if (this.editModalId === "story-modal") {
        this.editModal?.classList.remove("crop");
      } else {
        this.editModal?.close();
      }
    });

    this.editResetButton?.addEventListener("click", () => {
      if (!this.cropper) return;
      this.cropper.reset();
      if (this.rotateSlider) {
        this.rotateSlider.value = 0;
      }
      if (this.degreeElement) {
        this.degreeElement.textContent = "0";
      }
    });

    this.editDoneButton?.addEventListener("click", async () => {
      try {
        const editedData = await this.getEditedFile();
        if (editedData) {
          if (this.onEdit) {
            await this.onEdit(editedData);
          } else {
            const dataTransfer = new DataTransfer();
            dataTransfer.items.add(editedData.file);
            this.onFileChange?.(dataTransfer.files);
            if (this.inputField?.hasAttribute("name")) {
              this.inputField.files = dataTransfer.files;
            }
          }
        }
      } catch (error) {
        console.error("Error processing edited file:", error);
      }
    });

    this.rotateSlider?.addEventListener("input", (event) => {
      if (!this.cropper) return;
      const value = event.target.value;
      this.cropper.rotateTo(value);
      if (this.degreeElement) {
        this.degreeElement.textContent = value;
      }
    });

    this.fitHorizontalButton?.addEventListener("click", () => {
      if (!this.cropper) return;
      const containerData = this.cropper.getContainerData();
      const cropBoxData = this.cropper.getCropBoxData();
      this.cropper.setCropBoxData({
        width: containerData.width,
        left: 0,
        height: cropBoxData.height,
      });
    });

    this.fitVerticalButton?.addEventListener("click", () => {
      if (!this.cropper) return;
      const containerData = this.cropper.getContainerData();
      const cropBoxData = this.cropper.getCropBoxData();
      this.cropper.setCropBoxData({
        height: containerData.height,
        top: 0,
        width: cropBoxData.width,
      });
    });

    this.rotate90Button?.addEventListener("click", () => {
      if (!this.cropper) return;
      const currentRotation = parseInt(this.rotateSlider?.value || 0);
      const newRotation = (currentRotation + 90) % 360;
      this.cropper.rotateTo(newRotation);
      if (this.rotateSlider) {
        this.rotateSlider.value = newRotation;
      }
      if (this.degreeElement) {
        this.degreeElement.textContent = newRotation;
      }
    });

    // Add event listener for modal close to prevent cropper destruction
    if (this.editModal && this.editModalId !== "story-modal") {
      this.editModal.addEventListener("close", (e) => {
        // Don't destroy the cropper when the modal is closed
        e.preventDefault();
        this.editModal.close();
      });
    }
  }

  async processEditedImage() {
    if (this.isProcessing) {
      return;
    }

    this.isProcessing = true;

    try {
      const editedData = await this.getEditedFile();
      if (editedData) {
        if (this.onEdit) {
          await this.onEdit(editedData);
        } else {
          const dataTransfer = new DataTransfer();
          dataTransfer.items.add(editedData.file);
          this.onFileChange?.(dataTransfer.files);
          if (this.inputField?.hasAttribute("name")) {
            this.inputField.files = dataTransfer.files;
          }
        }
      }
    } catch (error) {
      console.error("Error processing edited file:", error);
      throw error;
    } finally {
      this.isProcessing = false;
    }
  }

  processImage(e) {
    const file = e.target.files[0];
    if (!file) {
      return;
    }

    if (!file.type.includes("image")) {
      e.target.value = "";
      alert("Please upload an image");
      return;
    }

    // Only destroy previous cropper if we're loading a new image
    if (this.cropper && this.file !== file) {
      this.cropper.destroy();
      this.cropper = null;
    }

    this.file = file;
    const objectUrl = URL.createObjectURL(file);
    this.editImageWorkspace.src = objectUrl;

    this.editImageWorkspace.onload = () => {
      // Only initialize cropper if it doesn't exist
      if (!this.cropper) {
        this.initCropper();
      }
      if (this.editModalId === "story-modal") {
        this.editModal?.classList.add("crop");
      } else {
        this.editModal?.showModal();
      }
    };

    this.editImageWorkspace.onerror = () => {
      if (this.cropper) {
        this.cropper.destroy();
        this.cropper = null;
      }
      URL.revokeObjectURL(this.editImageWorkspace.src);
      this.editImageWorkspace.src = "";
    };
  }

  initCropper() {
    if (!window.Cropper) {
      console.error("Cropper.js is not loaded");
      return;
    }

    try {
      this.cropper = new Cropper(this.editImageWorkspace, {
        aspectRatio: 250 / 250,
        viewMode: 0,
        rotatable: true,
        zoomable: true,
        movable: true,
        autoCropArea: 1,
        zoomOnWheel: false,
        dragMode: "move",
        ready: () => {
          console.log("Cropper initialized successfully");
        },
      });
    } catch (error) {
      console.error("Error initializing cropper:", error);
    }
  }

  // Method to be called externally to clean up resources
  destroy() {
    try {
      if (this.cropper) {
        this.cropper.destroy();
        this.cropper = null;
      }

      if (this.editImageWorkspace?.src) {
        URL.revokeObjectURL(this.editImageWorkspace.src);
        this.editImageWorkspace.src = "";
      }

      if (this.inputField) {
        this.inputField.value = "";
      }

      if (this.rotateSlider) {
        this.rotateSlider.value = 0;
      }

      if (this.degreeElement) {
        this.degreeElement.textContent = "0";
      }

      this.file = null;
      this.isProcessing = false;
      this.processingQueue = Promise.resolve();
    } catch (error) {
      console.error("Error during cleanup:", error);
    }
  }

  getEditedFile() {
    return new Promise((resolve, reject) => {
      if (!this.cropper) {
        reject(new Error("Cropper is not initialized"));
        return;
      }

      try {
        const croppedCanvas = this.cropper.getCroppedCanvas({
          imageSmoothingQuality: "high",
        });

        if (!croppedCanvas) {
          reject(new Error("Failed to get cropped canvas"));
          return;
        }

        const fileType = this.file.type.includes("svg")
          ? "image/png"
          : this.file.type;
        const fileName = this.file.name.includes("svg")
          ? `${this.file.name.split(".").shift()}.png`
          : this.file.name;

        croppedCanvas.toBlob((blob) => {
          if (!blob) {
            reject(new Error("Failed to create blob"));
            return;
          }

          const editedImageFile = new File([blob], fileName, {
            type: fileType,
          });

          const blobUrl = URL.createObjectURL(blob);
          const base64Url = croppedCanvas.toDataURL(fileType);

          resolve({
            file: editedImageFile,
            size: blob.size,
            url: blobUrl,
            base64Url: base64Url,
            name: fileName,
          });
        }, fileType);
      } catch (error) {
        reject(error);
      }
    });
  }
}

function formatISODate(isoDate) {
  const now = new Date();
  const date = new Date(isoDate);

  if (isNaN(date.getTime())) {
    throw new Error("Invalid ISO date format");
  }

  const seconds = Math.floor((now - date) / 1000);

  if (seconds < 60) {
    return `${seconds} seconds ago`;
  }

  const minutes = Math.floor(seconds / 60);
  if (minutes < 60) {
    const remainingSeconds = seconds % 60;
    return `${minutes} minute${minutes > 1 ? "s" : ""}${
      remainingSeconds
        ? ` and ${remainingSeconds} second${remainingSeconds > 1 ? "s" : ""}`
        : ""
    } ago`;
  }

  const hours = Math.floor(minutes / 60);
  if (hours < 24) {
    const remainingMinutes = minutes % 60;
    return `${hours} hour${hours > 1 ? "s" : ""}${
      remainingMinutes
        ? ` and ${remainingMinutes} minute${remainingMinutes > 1 ? "s" : ""}`
        : ""
    } ago`;
  }

  const days = Math.floor(hours / 24);
  if (days < 7) {
    return `${days} day${days > 1 ? "s" : ""} ago`;
  }

  const weeks = Math.floor(days / 7);
  if (weeks < 2) {
    const remainingDays = days % 7;
    return `${weeks} week${weeks > 1 ? "s" : ""}${
      remainingDays
        ? ` and ${remainingDays} day${remainingDays > 1 ? "s" : ""}`
        : ""
    } ago`;
  }

  if (weeks < 4) {
    return `${weeks} week${weeks > 1 ? "s" : ""} ago`;
  }

  const months = Math.floor(days / 30.44); // Average days in a month
  if (months < 12) {
    return `${months} month${months > 1 ? "s" : ""} ago`;
  }

  const years = Math.floor(months / 12);
  return `${years} year${years > 1 ? "s" : ""} ago`;
}

function formatDate(isoDate) {
  const date = new Date(isoDate);

  return date.toLocaleDateString("en-GB", {
    day: "2-digit",
    month: "short",
    year: "numeric",
  });
}

function formatTime(isoString) {
  const date = new Date(isoString);
  let hours = date.getHours();
  const minutes = date.getMinutes();
  const amPm = hours >= 12 ? "pm" : "am";

  hours = hours % 12 || 12;
  const formattedMinutes = minutes.toString().padStart(2, "0");

  return `${hours}:${formattedMinutes}${amPm}`;
}

// Format file size
function formatFileSize(bytes) {
  if (bytes === 0) return "0 KB";

  const sizes = ["Bytes", "KB", "MB", "GB"];
  const i = Math.floor(Math.log(bytes) / Math.log(1024));
  const size = (bytes / Math.pow(1024, i)).toFixed(1);

  return `${size} ${sizes[i]}`;
}

// Generate IDs
function generateUniqueId(type) {
  const timestamp = Date.now();
  const randomNum = Math.floor(Math.random() * 10000);

  switch (type) {
    case "menuCategory":
      return `MC-${timestamp}-${randomNum}`;
    case "dish":
      return `DISH-${timestamp}-${randomNum}`;
    case "sideItem":
      return `SIDE-${timestamp}-${randomNum}`;
    case "sideItemCategory":
      return `SIC-${timestamp}-${randomNum}`;
    default:
      throw new Error("Invalid type provided for ID generation.");
  }
}

// Validate email
function validateEmail(email) {
  const emailRegex = /^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;
  return emailRegex.test(email);
}
