<?php

namespace App\helpers;

class NigerianPhone
{
    private string $phoneNumber;

    private array $prefixesByNetwork = [
        'Glo' => ['0805', '0705', '0905', '0807', '0815', '0811'],
        'MTN' => ['0806', '0803', '0816', '0813', '0810', '0814', '0903', '0906', '0703', '0706', '0704', '0911'],
        'Airtel' => ['0802', '0902', '0701', '0808', '0708', '0812', '0907'],
        '9mobile' => ['0809', '0909', '0817', '0818', '0908'],
        'Ntel' => ['0804'],
        'Smile' => ['0702'],
    ];

    private static $networkByPrefix = [
        '0803','0806','0703','0706','0813','0816','0810','0814','0903','0906','0805','0807','0705','0815','0811', '0911', '0905','0802','0808','0708','0704','0812','0701','0901','0902', '0907', '0904','0809','0818','0817','0908','0909','0804','0702','07028','07029','0819','07025','07026','07027','0709','0707'
    ];

    public function __construct(string $phoneNumber)
    {
        $this->phoneNumber = $this->sanitizePhoneNumber($phoneNumber);
    }

    public function isValid(): bool
    {
        return isset($this->phoneNumber) && is_numeric($this->phoneNumber) && in_array(substr($this->phoneNumber, 0, 4), self::$networkByPrefix);
    }

    public function getNetwork(): ?string
    {
        foreach ($this->prefixesByNetwork as $network => $prefixes) {
            if (in_array(substr($this->phoneNumber, 0, 4), $prefixes)) {
                return $network;
            }
        }
        return 'Unknown';
    }

    public function formatted(): string
    {
        return '+234' . substr($this->phoneNumber, -10);
    }

    public function official(): string
    {
        $prefix = substr($this->phoneNumber, 0, 3);
        $middle = substr($this->phoneNumber, 3, 3);
        $suffix = substr($this->phoneNumber, 6, 5);
        return "($prefix) $middle-$suffix";
    }

    public function isNetwork(string $networkName): bool {
        return $this->isValid() && in_array(substr($this->phoneNumber, 1, 3), $this->prefixesByNetwork[$networkName]);
    }

    public function isMtn(): bool {
        return $this->isNetwork('MTN');
    }

    public function isGlo(): bool {
        return $this->isNetwork('Glo');
    }

    public function isAirtel(): bool {
        return $this->isNetwork('Airtel');
    }

    public function is9mobile(): bool {
        return $this->isNetwork('9mobile');
    }

    public function isSmile(): bool
    {
        return $this->isNetwork('Smile');
    }

    public function isMultilinks(): bool
    {
        return $this->isNetwork('Multilinks');
    }

    public function isVisafone(): bool
    {
        return $this->isNetwork('Visafone');
    }

    public function isNtel(): bool
    {
        return $this->isNetwork('Ntel');
    }

    public function isStarcomms(): bool
    {
        return $this->isNetwork('Starcomms');
    }

    private function sanitizePhoneNumber(string $phoneNumber): string
    {
        // Remove all non-numeric characters from the phone number
        $phoneNumber = preg_replace('/[^0-9]/', '', $phoneNumber);

        // Check if the phone number starts with '234' or '+234' and replace with '0'
        if (substr($phoneNumber, 0, 3) === '234') {
            $phoneNumber = '0' . substr($phoneNumber, 3);
        } elseif (substr($phoneNumber, 0, 4) === '+234') {
            $phoneNumber = '0' . substr($phoneNumber, 4);
        }

        // Return the sanitized phone number
        return $phoneNumber;
    }

}
