<?php
declare(strict_types=1);

namespace App\Helpers;

class PasswordHandler
{
    const DEFAULT_ALGORITHM = PASSWORD_BCRYPT;
    const BCRYPT_COST = 12; // Adjust the cost factor based on your needs

    /**
     * Hash a password using the specified algorithm.
     *
     * @param string $password
     * @param int $algorithm
     * @return string|false
     */
    public static function hashPassword($password, $algorithm = self::DEFAULT_ALGORITHM)
    {
        $options = [];

        if ($algorithm === PASSWORD_BCRYPT) {
            $options['cost'] = self::BCRYPT_COST;
        }

        return password_hash($password, $algorithm, $options);
    }

    /**
     * Verify if the given password matches the hashed password.
     *
     * @param string $password
     * @param string $hashedPassword
     * @return bool
     */
    public static function verifyPassword($password, $hashedPassword)
    {
        return password_verify($password, $hashedPassword);
    }

    /**
     * Check if a password needs to be rehashed.
     *
     * @param string $hashedPassword
     * @param int $algorithm
     * @return bool
     */
    public static function needsRehash($hashedPassword, $algorithm = self::DEFAULT_ALGORITHM): bool
    {
        return password_needs_rehash($hashedPassword, $algorithm);
    }
}
